package com.yaoyaozw.customer.handler;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.yaoyaozw.customer.common.R;
import com.yaoyaozw.customer.constants.ApiResultConstant;
import com.yaoyaozw.customer.constants.CustomerCommonConstant;
import com.yaoyaozw.customer.entity.AuthorizerExpandInfo;
import com.yaoyaozw.customer.entity.AuthorizerInfo;
import com.yaoyaozw.customer.entity.ReferralEntity;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;

/**
 * 番茄书城推广链接处理器
 *
 * @author system
 */
@Slf4j
@Component
public class TomatoMpReferralHandler extends MpReferralHandler {

    private static final String GET_BOOK_ITEM = "get_book_item_id";
    private static final String LANDING_PAGE = "landing_page";
    private static final String FROM_APPID = "from_oa_app_id";
    private static final String GET_BOOK_ITEM_VAL = "get_book_item_id=true";
    private static final String LANDING_PAGE_VAL = "landing_page=reader";

    @Override
    public ReferralEntity queryMpReferral(ReferralEntity referralEntity, AuthorizerInfo authorizerInfo) {
        // 根据appid查询authorizer_expand_info表（1条）
        String appid = authorizerInfo.getAppid();
        LambdaQueryWrapper<AuthorizerExpandInfo> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(AuthorizerExpandInfo::getAuthorizerAppid, appid);
        queryWrapper.last("LIMIT 1");
        AuthorizerExpandInfo authorizerExpandInfo = authorizerExpandInfoService.getOne(queryWrapper);
        if (authorizerExpandInfo == null) {
            throw new IllegalArgumentException("未找到对应的公众号扩展信息，appid: " + appid);
        }
        referralEntity.setMpAppId(authorizerExpandInfo.getMpAppid());

        // 处理参数：设置accountId为mpDistributorId
        referralEntity.setInfoId(authorizerInfo.getId());
        referralEntity.setAccountName(authorizerInfo.getNickName());
        if (CustomerCommonConstant.BOOK_NEWS_TYPE.equals(referralEntity.getNewsType())) {
            // 参数设置 mediaSource
            referralEntity.setMediaSource("2");
        }
        if (CustomerCommonConstant.ACTIVITY_NEWS_TYPE.equals(referralEntity.getNewsType())) {
            // 活动不使用小程序的分销商Id，使用公众号原分销商Id
            referralEntity.setAccountId(authorizerInfo.getAccountId());
            if (StringUtils.isBlank(referralEntity.getActivityTitle())) {
                // 构造活动标题
                String activityTitle = "";
                if (referralEntity.getRechargeAmount() != null && BigDecimal.ZERO.compareTo(referralEntity.getRechargeAmount()) != 0) {
                    activityTitle += "充" + referralEntity.getRechargeAmount().toPlainString();
                }
                if (referralEntity.getGiftAmount() != null && referralEntity.getGiftAmount() > 0) {
                    activityTitle += "送" + referralEntity.getGiftAmount() + "书币";
                }
                referralEntity.setActivityTitle(activityTitle);
            }
        } else {
            // 活动之外的，使用小程序的分销商Id（书籍、常用链接）
            referralEntity.setAccountId(authorizerExpandInfo.getMpDistributorId());
        }
        referralEntity.setMiniProgramQuery(Boolean.TRUE);
        // 调用接口获取原生链接
        R r = referralFeignClient.productReferral(referralEntity);
        if (!r.getCode().equals(ApiResultConstant.SUCCESS_CODE)) {
            throw new RuntimeException("获取链接失败: " + r.getMessage());
        }
        String res = r.getData("storeReferral", new TypeReference<String>() {});
        JSONObject jsonObject = JSON.parseObject(res);

        // 处理链接
        constructMpReferral(referralEntity, authorizerInfo, authorizerExpandInfo, jsonObject);
        return referralEntity;
    }

    @Override
    public void constructMpReferral(ReferralEntity referralEntity, AuthorizerInfo authorizerInfo, AuthorizerExpandInfo authorizerExpandInfo, JSONObject responseJson) {
        if (referralEntity == null) {
            return;
        }
        // 处理链接：添加番茄小程序所需参数
        referralEntity.setPromoteId(responseJson.getString("promoteId"));
        String referral = responseJson.getString("referral");
        referralEntity.setReferral(referral);
        // 判断原始链接是不是http链接
        if (StringUtils.isNotBlank(referral) && isHttpPath(referral)) {
            referralEntity.setExtraHttpUrl(referral);
        }
        // 判断是不是活动类型，如果是活动类型，因为创建成功之后接口不返回小程序链接，所以还需要查询链接列表接口，获取小程序链接
        if (CustomerCommonConstant.ACTIVITY_NEWS_TYPE.equals(referralEntity.getNewsType())) {
            // 根据 promoteId 查询番茄链接详情，番茄活动接口创建链接返回的是http链接，需要查询接口获取其小程序 /page 链接
            R r = referralFeignClient.queryActivityInfo(referralEntity.getAccountId(), referralEntity.getPromoteId());
            if (!r.getCode().equals(ApiResultConstant.SUCCESS_CODE)) {
                throw new RuntimeException("获取活动链接详情失败: " + r.getMessage());
            }
            String res = r.getData("activityData", new TypeReference<String>() {});
            JSONObject jsonObject = JSON.parseObject(res);
            referral = jsonObject.getString("url");
        }
        if (StringUtils.isNotBlank(referral) && isMiniProgramPath(referral)) {
            // 添加 get_book_item_id=true 参数
            if (!referral.contains(GET_BOOK_ITEM)) {
                referral = referral + "&" + GET_BOOK_ITEM_VAL;
            }
            // 添加 from_oa_app_id={flagId} 参数
            if (!referral.contains(FROM_APPID)) {
                referral = referral + "&" + FROM_APPID + "=" + authorizerExpandInfo.getFlagId();
            }

            // 添加 landing_page=reader 参数
            if (!referral.contains(LANDING_PAGE) && !CustomerCommonConstant.USUAL_LINK_NEWS_TYPE.equals(referralEntity.getNewsType())) {
                referral = referral + "&" + LANDING_PAGE_VAL;
            }
            // 常用链接换掉scene
            if (CustomerCommonConstant.USUAL_LINK_NEWS_TYPE.equals(referralEntity.getNewsType())) {
                referral = referral.replace("scene=0", "scene=1");
            }
        }
        referralEntity.setMpAppId(authorizerExpandInfo.getMpAppid());
        referralEntity.setMpPath(referral);
        // 暂时将小程序地址也赋值到跳转链接地址上
        referralEntity.setReferral(referral);
    }
}

