package com.yaoyaozw.customer.service.wechat.service;


import com.alibaba.fastjson.JSONObject;
import com.yaoyaozw.customer.constants.CustomerCommonConstant;
import com.yaoyaozw.customer.constants.RabbitCommonNameConstant;
import com.yaoyaozw.customer.dto.customer.CustomerMessageTransferDTO;
import com.yaoyaozw.customer.entity.AuthorizerToken;
import com.yaoyaozw.customer.entity.CustomerDelayPublish;
import com.yaoyaozw.customer.entity.CustomerGraphics;
import com.yaoyaozw.customer.service.AuthorizerTokenService;
import com.yaoyaozw.customer.service.wechat.entity.WeChatResponseEntity;
import com.yaoyaozw.customer.service.wechat.entity.customerRequest.WeChatCustomerRequestEntity;
import com.yaoyaozw.customer.vo.customer.CrowdPackageUserVO;
import com.yaoyaozw.customer.vo.customer.CustomerDelayItemVO;
import com.yaoyaozw.customer.vo.referral.ReferralEntityVo;
import org.apache.commons.lang3.StringUtils;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.AsyncResult;
import org.springframework.stereotype.Service;
import org.springframework.web.util.UriComponentsBuilder;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

@Service
public class WeChatServiceImpl implements WeChatService{

    private String customerPath="https://api.weixin.qq.com/cgi-bin/message/custom/send";

    private static final String CUSTOMER_TEXT="text";

    private static final String CUSTOMER_NEWS="news";

    private static final int THREAD_SIZE=1000;

    private static final Integer SUCCESS_CODE=0;

    private static final Integer EXPIRED_CODE=40001;

    private static final Integer FORBID_TIME=6;

    private static final String ACCESS_TOKEN="access_token";

    private static final ThreadPoolExecutor EXECUTOR = new ThreadPoolExecutor(20, 100, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<>(10),new ThreadPoolExecutor.CallerRunsPolicy());

    private final static String ACCESS_TOKEN_REDIS_KEY = "AUTH_ACCESS_TOKEN";

    @Autowired
    private RedisTemplate<String,Object> redisTemplate;


    @Autowired
    private WeChatRestService weChatRestService;

    @Autowired
    private RabbitTemplate rabbitTemplate;

    @Async("delayCustomerExecutor")
    @Override
    public Future<CustomerDelayPublish> sendCustomerDelayMessage( String appid,String token , CustomerDelayPublish user,
                                                                  Map<Integer,List<CustomerDelayItemVO>> delaySortMap,
                                                                  Set<CustomerDelayItemVO> needUpdateVoList) {
        //获取当前小时数
        Calendar instance = Calendar.getInstance();
        int i = instance.get(Calendar.HOUR_OF_DAY);
        //早上0-6点不发
        if (i>=FORBID_TIME){
            //token
            UriComponentsBuilder uriComponentsBuilder = UriComponentsBuilder.fromHttpUrl(customerPath).queryParam(ACCESS_TOKEN,token);
            //找到延时序列
            List<CustomerDelayItemVO> sendCustomerDelays = delaySortMap.get(user.getCustomerSort());

            if (sendCustomerDelays!=null){
                //找付费类型
                List<CustomerDelayItemVO> collect = sendCustomerDelays.stream().filter(a -> a.getIsPay().equals(-1)||user.getPayType().equals(a.getIsPay()) ).collect(Collectors.toList());

                if (!collect.isEmpty()){

                    CustomerDelayItemVO sendCustomerDelay = collect.get(0);

                    needUpdateVoList.add(sendCustomerDelay);
                    //构建请求参数
                    WeChatCustomerRequestEntity customerRequest = buildCustomerRequest(sendCustomerDelay);

                    if (customerRequest!=null){

                        customerRequest.setTouser(user.getOpenId());

                        try {
                            WeChatResponseEntity response=(WeChatResponseEntity)weChatRestService.httpPostRequest(uriComponentsBuilder,  customerRequest,WeChatResponseEntity.class);

                            if(SUCCESS_CODE.equals(response.getErrcode())){
                                //发送成功计数
                                sendCustomerDelay.updateSendNum();

                            }else if(EXPIRED_CODE.equals(response.getErrcode()) ){

                                Object tokenObject = redisTemplate.opsForHash().get(ACCESS_TOKEN_REDIS_KEY, appid);

                                if (tokenObject!=null){

                                    return sendCustomerDelayMessage(appid,tokenObject.toString(),user,delaySortMap,needUpdateVoList);
                                }
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            }
        }
        //用户首次活跃时间
        Long subscribeTimestamp = user.getFirstActive()!=null?user.getFirstActive().getTime():user.getGmtCreate().getTime();

        Set<Integer> sortSet = delaySortMap.keySet();
        //找大于当前序列的最小序列
        Integer newSort =sortSet.stream().filter(a->a>user.getCustomerSort()).min(Comparator.naturalOrder()).orElse(-1);

        user.setCustomerSort(newSort);

        CustomerDelayItemVO customerDelayItemVO = delaySortMap.get(newSort)!=null?delaySortMap.get(newSort).get(0):null;
        //下次排期
        if (customerDelayItemVO!=null&&customerDelayItemVO.getTimeInterval()!=null){
            //时间戳舍秒和毫秒
            user.setCustomerPublish(new Date((subscribeTimestamp/(60*1000))*60*1000+customerDelayItemVO.getTimeInterval()));
        }
        return new AsyncResult<>(user);
    }


    @Override
    public void sendCustomerMessage(String appid,  CustomerGraphics customerGraphics, List<CrowdPackageUserVO> openidList,
                                    List<ReferralEntityVo> referralEntityVo) {
        //构建请求参数（文本/图文）
        WeChatCustomerRequestEntity customerRequest = buildCustomerRequest(customerGraphics,referralEntityVo);

        if (customerRequest!=null){
            //计数器
            CountDownLatch latch = new CountDownLatch(openidList.size());

            for (int i = 0; i < openidList.size(); i+=THREAD_SIZE) {

                List<CrowdPackageUserVO> subOpenidList = openidList.subList(i, Math.min(i + THREAD_SIZE, openidList.size()));

                EXECUTOR.execute(()->{

                    for (CrowdPackageUserVO crowdPackageUserVO : subOpenidList) {

                        try {
                            CustomerMessageTransferDTO customerMessageTransferDTO = new CustomerMessageTransferDTO();
                            customerMessageTransferDTO.setAppid(appid);
                            customerMessageTransferDTO.setCustomerGraphicsId(customerGraphics.getId());
                            customerMessageTransferDTO.setOpenid(crowdPackageUserVO.getOpenId());
                            customerMessageTransferDTO.setCustomerRequestEntity(customerRequest);

                            rabbitTemplate.convertAndSend(RabbitCommonNameConstant.CUSTOMER_SERVICE_EXCHANGE, RabbitCommonNameConstant.CUSTOMER_MESSAGE_ROUTE_KEY,JSONObject.toJSONString(customerMessageTransferDTO));
                        } finally {
                            latch.countDown();
                        }
                    }
                });
            }
            try {
                latch.await();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }


    /**
     * 延时客服参数构建
     * @param customerDelay
     * @return
     */
    private WeChatCustomerRequestEntity buildCustomerRequest(CustomerDelayItemVO customerDelay) {

        //类型判断
        if (CUSTOMER_TEXT.equals(customerDelay.getType())){
            //文本类
            return new WeChatCustomerRequestEntity(CUSTOMER_TEXT,customerDelay.getContent());

        }else if (CUSTOMER_NEWS.equals(customerDelay.getType())){
            //链接生成失败不发
            if (CustomerCommonConstant.ERROR_LABEL.equals(customerDelay.getSourceUrl())|| StringUtils.isBlank(customerDelay.getSourceUrl())){

                return null;
            }
            //图文类
            return new WeChatCustomerRequestEntity(CUSTOMER_NEWS,customerDelay.getExtendTitle(),customerDelay.getContent(),customerDelay.getSourceUrl(),customerDelay.getCoverUrl());
        }
        return null;
    }



    /**
     * 定时客服参数构建
     * @param customerGraphics
     * @param urlList
     * @return
     */
    private WeChatCustomerRequestEntity buildCustomerRequest(CustomerGraphics customerGraphics,List<ReferralEntityVo> urlList){
        //类型判断
        if (CUSTOMER_TEXT.equals(customerGraphics.getType())){
            //文本类客服，需要判断替换链接的个数
            Map<Integer, String> sortReferral = urlList!=null&&!urlList.isEmpty()?urlList.stream().collect(Collectors.toMap(ReferralEntityVo::getSort, ReferralEntityVo::getReferral, (v1, v2) -> v2)):new HashMap<>(1);

            if (customerGraphics.getReferralSize()!=null&&customerGraphics.getReferralSize().equals(sortReferral.size())){

                String content = customerGraphics.getContent();

                for (Map.Entry<Integer, String> replaceReferral : sortReferral.entrySet()) {

                    Integer sort = replaceReferral.getKey();

                    String url = replaceReferral.getValue();
                    //替换占位符
                    content=content.replace(CustomerCommonConstant.CUSTOMER_TEXT_URL_PLACEHOLDER + sort, url);
                }
                return new WeChatCustomerRequestEntity(CUSTOMER_TEXT,content);

            }else{

                return null;
            }
        }else if (CUSTOMER_NEWS.equals(customerGraphics.getType())){

            if (urlList!=null&&!urlList.isEmpty()){
                return new WeChatCustomerRequestEntity(CUSTOMER_NEWS,customerGraphics.getExtendTitle(),customerGraphics.getContent(),urlList.get(0).getReferral(),customerGraphics.getCoverUrl());
            }
        }
        return null;
    }

}
