package com.yaoyaozw.customer.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yaoyaozw.customer.common.BaseResult;
import com.yaoyaozw.customer.common.GenericsResult;
import com.yaoyaozw.customer.components.CustomerServiceCommonAsyncComponent;
import com.yaoyaozw.customer.components.TokenManager;
import com.yaoyaozw.customer.constants.CrowdPackageCommonConstant;
import com.yaoyaozw.customer.dto.crowd.CrowdPackageQueryDTO;
import com.yaoyaozw.customer.dto.crowd.CrowdPackageConditionDTO;
import com.yaoyaozw.customer.entity.CrowdPackage;
import com.yaoyaozw.customer.entity.CrowdPackageCondition;
import com.yaoyaozw.customer.entity.CrowdPackageConditionMatch;
import com.yaoyaozw.customer.enums.CrowdPackageConditionEnum;
import com.yaoyaozw.customer.mapper.KanbanCommonMapper;
import com.yaoyaozw.customer.mapper.MaterialCrowdPackageMapper;
import com.yaoyaozw.customer.mapper.RegisterUserEntityMapper;
import com.yaoyaozw.customer.service.CrowdPackageConditionMatchService;
import com.yaoyaozw.customer.service.CrowdPackageConditionService;
import com.yaoyaozw.customer.service.CrowdPackageService;
import com.yaoyaozw.customer.vo.kanban.CommonOptionResponseVO;
import com.yaoyaozw.customer.vo.crowd.*;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author darker
 * @date 2022/9/16 12:03
 */
@Service
public class CrowdPackageServiceImpl extends ServiceImpl<MaterialCrowdPackageMapper, CrowdPackage> implements CrowdPackageService {

    private final static Logger LOCAL_LOG = LoggerFactory.getLogger(CrowdPackageServiceImpl.class);

    @Autowired
    private CrowdPackageConditionService conditionService;
    @Autowired
    private KanbanCommonMapper kanbanCommonMapper;
    @Autowired
    private TokenManager tokenManager;
    @Autowired
    private CrowdPackageConditionMatchService matchService;
    @Autowired
    private RegisterUserEntityMapper userEntityMapper;
    @Autowired
    private RedisTemplate redisTemplate;
    @Autowired
    private CustomerServiceCommonAsyncComponent asyncComponent;


    @Override
    public GenericsResult<CrowdPackageCommonIdVO> insertCrowdPackage(Long id, String name) {
        // 根据是否传了主键判断是创建还是更新
        boolean isCreate = ObjectUtil.isNull(id);
        // 获取操作人
        Long userId = tokenManager.getUserIdFromToken();
        CrowdPackage crowdPackage = new CrowdPackage(id, name);
        crowdPackage.handleInfo(new Date(), userId, isCreate);
        if (ObjectUtil.isNotNull(name)) {
            // 判断是否有重名
            QueryWrapper<CrowdPackage> nameRepeatWrapper = new QueryWrapper<CrowdPackage>().eq("package_name", name).ne("id", id);
            int count = super.count(nameRepeatWrapper);
            if (count != 0) {
                LOCAL_LOG.info("人群包名: '{}' 已存在", name);
                return new GenericsResult<>(false, "人群包名已存在!");
            }
        }
        // 执行保存
        boolean result = super.saveOrUpdate(crowdPackage);

        // TODO: 2022/9/26 添加人群包人群的更新
        if (result && ObjectUtil.isNotNull(name)) {
            asyncComponent.addMatchUserIntoPackage(id, true);
        }
        
        return result ? new GenericsResult<>(new CrowdPackageCommonIdVO(crowdPackage.getId(), null)) : new GenericsResult<>(false, "新增人群包失败!");
    }

    @Override
    public GenericsResult<CrowdPackageCommonIdVO> insertConditionIntoPackage(CrowdPackageConditionDTO conditionDto) {
        CrowdPackageCommonIdVO crowdPackageCommonIdVo = new CrowdPackageCommonIdVO();
        Long packageId = conditionDto.getPackageId();
        crowdPackageCommonIdVo.setPackageId(packageId);

        // 构造关联表的相关数据
        CrowdPackageConditionMatch match = new CrowdPackageConditionMatch();
        match.setId(conditionDto.getMatchId());
        match.setPackageId(crowdPackageCommonIdVo.getPackageId());
        match.setConditionId(conditionDto.getConditionId());
        // 构造条件数据
        try {
            constructOperator(match, conditionDto);
        } catch (Exception e) {
            LOCAL_LOG.info("构造运算条件异常");
            return new GenericsResult<>(false, e.getMessage());
        }
        crowdPackageCommonIdVo.setConditionDescription(match.getOperatorDescription());
        // 保存关联关系
        boolean matchSaveResult = matchService.saveOrUpdate(match);
        if (!matchSaveResult) {
            return new GenericsResult<>(false, "向人群包保存条件失败!");
        }
        // 构造返回数据
        crowdPackageCommonIdVo.setLinkedId(match.getId());
        return new GenericsResult<>(crowdPackageCommonIdVo);
    }

    @Override
    public GenericsResult<CrowdPackageDetailVO> getPackageInfo(Long id) {
        LOCAL_LOG.info("获取详情");
        CrowdPackage byId = super.getById(id);
        if (ObjectUtil.isNull(byId)) {
            return new GenericsResult<>(false, "无法获取主体数据");
        }
        CrowdPackageDetailVO vo = new CrowdPackageDetailVO(id, byId.getPackageName());
        // 获取条件数据
        List<CrowdPackageConditionMatch> conditionMatchList = matchService.list(new QueryWrapper<CrowdPackageConditionMatch>().eq("package_id", id));
        JSONArray jsonArray = JSONUtil.parseArray(conditionMatchList);
        List<CrowdPackageConditionMatchVO> parseConditionList = jsonArray.toList(CrowdPackageConditionMatchVO.class);
        vo.setConditionList(parseConditionList);
        return new GenericsResult<>(vo);
    }

    @Override
    public GenericsResult<List<CrowdPackageListVO>> pageList(CrowdPackageQueryDTO queryDto) {
        List<CrowdPackageListVO> pageList = baseMapper.getPageList(queryDto);
        if (CollectionUtil.isEmpty(pageList)) {
            return new GenericsResult<>(true, "暂无数据");
        }

        return new GenericsResult<>(pageList);
    }

    @Override
    public BaseResult removeCrowdPackage(Long id) {
        return super.removeById(id) ? new BaseResult().success() : new BaseResult().error("删除异常!");
    }

    @Override
    public BaseResult removeConditionFromPackage(Long linkedId) {
        baseMapper.removeCondition(linkedId);
        return new BaseResult().success();
    }

    @Override
    public GenericsResult<List<CrowdPackageConditionVO>> getAllConditions() {
        LOCAL_LOG.info("获取条件列表");
        List<CrowdPackageCondition> conditionList = conditionService.list();
        // 将结果转换成返回结果
        LOCAL_LOG.info("转换条件数据");
        List<CrowdPackageConditionVO> conditionVoList = conditionList.stream().map(item -> {
            CrowdPackageConditionVO vo = new CrowdPackageConditionVO();
            BeanUtils.copyProperties(item, vo);
            return vo;
        }).collect(Collectors.toList());
        // 封装返回结果
        LOCAL_LOG.info("返回条件结果");
        return new GenericsResult<>(conditionVoList);
    }

    @Override
    public GenericsResult<ConditionOptionResponseVO> getOptions(Long conditionId) {
        // 查出该条件的数据
        CrowdPackageCondition conditionItem = conditionService.getById(conditionId);
        if (ObjectUtil.isNull(conditionItem)) {
            return new GenericsResult<>(Boolean.FALSE, CrowdPackageCommonConstant.CANT_FIND_ITEM);
        }
        if (ObjectUtil.isNull(conditionItem.getSourceBaseService())) {
            return new GenericsResult<>(Boolean.FALSE, CrowdPackageCommonConstant.CANT_FIND_DATA_BASE_CONFIG);
        }
        // 根据条件配置判断查询的数据库
        List<CommonOptionResponseVO> result;
        if (CrowdPackageCommonConstant.KANBAN_DATA_BASE.equals(conditionItem.getSourceBaseService())) {
            // 是看板的数据源
            result = kanbanCommonMapper.getConditionOptionByConfig(conditionItem);
        } else if (CrowdPackageCommonConstant.MATERIAL_DATA_BASE.equals(conditionItem.getSourceBaseService())) {
            // 是运营系统的数据源
            result = baseMapper.getConditionOptionByConfig(conditionItem);
        } else {
            return new GenericsResult<>(Boolean.FALSE, CrowdPackageCommonConstant.UNKNOWN_DATA_BASE_CONFIG);
        }

        return new GenericsResult<>(new ConditionOptionResponseVO(conditionItem.getFrontType(), result));
    }


    @Override
    public BaseResult updateUserPackageBelong(String openId) {
        List<CrowdPackage> packageList = super.list();
        StringBuilder packageConcatResult = new StringBuilder();

        boolean isFirst = true;
        for (CrowdPackage crowdPackage : packageList) {
            Boolean matchPackage = matchService.getUserPackageBelong(crowdPackage.getId(), openId);
            if (matchPackage) {
                // 用户符合人群包条件
                if (!isFirst) {
                    // 除了第一个，其他的要在数字前拼上逗号
                    packageConcatResult.append(",");
                }
                packageConcatResult.append(crowdPackage.getId());
                isFirst = false;
            }
        }
        String packageStr = packageConcatResult.toString();

        userEntityMapper.updateUserPackageBelong(openId, StringUtils.isBlank(packageStr) ? null : packageStr);

        LOCAL_LOG.info("用户openId: {} 人群包更新完成", openId);

        return new BaseResult().success();
    }


    @Override
    public BaseResult updateUserPackageBatch(List<String> openIdList) {
        for (String openId : openIdList) {
            this.updateUserPackageBelong(openId);
        }
        return new BaseResult().success();
    }

    /**
     * 构造操作符
     *
     * @param match        匹配
     * @param conditionDto 条件dto
     */
    private void constructOperator(CrowdPackageConditionMatch match, CrowdPackageConditionDTO conditionDto){
        LOCAL_LOG.info("构造运算表达式");
        CrowdPackageCondition conditionInfo = conditionService.getById(conditionDto.getConditionId());
        if (ObjectUtil.isNull(conditionInfo)) {
            throw new RuntimeException("无法获取条件");
        }
        match.setIsStatic(conditionInfo.getIsStatic());
        String conditionType = conditionInfo.getConditionType();
        LOCAL_LOG.info("当前条件类型: {}", conditionType);
        // 是需要比较的类型, 获取对应的比较符号的数据
        if (CrowdPackageCommonConstant.COMPARE_TYPE.equals(conditionType)) {
            handleCompareType(match, conditionDto, conditionInfo);
            return;
        }
        if (CrowdPackageCommonConstant.RANGE_IN.equals(conditionType)) {
            handleRangeIn(match, conditionDto, conditionInfo);
            return;
        }
        if (CrowdPackageCommonConstant.GROUP_OR.equals(conditionType)) {
            handleGroupOr(match, conditionDto, conditionInfo);
            return;
        }

        // 没有匹配的类型
        throw new RuntimeException("找不到匹配的条件类型!");

    }

    /**
     * 处理比较类型
     * @param match 匹配关系
     * @param conditionDto 请求条件
     * @param conditionInfo 条件信息
     */
    private void handleCompareType(CrowdPackageConditionMatch match, CrowdPackageConditionDTO conditionDto, CrowdPackageCondition conditionInfo){
        LOCAL_LOG.info("是需要比较的类型");

        String compareOperator = conditionDto.getCompareOperator();
        String conditionValue = conditionDto.getConditionValue();

        LOCAL_LOG.info("比较符: {}; 条件值: {}", compareOperator, conditionValue);
        match.setConditionOperator(compareOperator);
        match.setOperatorValue(conditionValue);
        CrowdPackageConditionEnum operatorInfo = CrowdPackageConditionEnum.getInfoByOperator(compareOperator);
        if (ObjectUtil.isNull(operatorInfo)) {
            throw new RuntimeException("无法获取比较类型");
        }

        String description = conditionInfo.getConditionName() + operatorInfo.getDescription() + conditionValue;
        LOCAL_LOG.info("条件描述: {}", description);

        // 判断是不是数字
        if (!CrowdPackageCommonConstant.NUMBER_COMPILE.matcher(conditionValue).matches()) {
            // 不是数字，添加引号，否则sql报错
            conditionValue = "'" + conditionValue + "'";
        }

        String expression = conditionInfo.getConditionKey() + " " + operatorInfo.getMeaning()  + " " + conditionValue;
        LOCAL_LOG.info("表达式拼接结果: {}", expression);

        match.setOperatorExpression(expression);
        match.setOperatorDescription(description);
    }

    /**
     * 处理范围类型
     * @param match 匹配关系
     * @param conditionDto 请求条件
     * @param conditionInfo 条件信息
     */
    private void handleRangeIn(CrowdPackageConditionMatch match, CrowdPackageConditionDTO conditionDto, CrowdPackageCondition conditionInfo) {
        // 范围条件
        LOCAL_LOG.info("是范围条件");
        List<String> multipleOptions = conditionDto.getMultipleOptions();
        if (CollectionUtil.isEmpty(multipleOptions)) {
            return;
        }
        StringBuilder expressionBuilder = new StringBuilder(conditionInfo.getConditionKey() + " in (");
        int location = 1;
        for (String multipleOption : multipleOptions) {
            boolean isLast = location == multipleOptions.size();
            expressionBuilder.append("'").append(multipleOption).append("'");
            if (!isLast) {
                // 给非最后一个元素拼接逗号
                expressionBuilder.append(", ");
            }
            location += 1;

        }
        expressionBuilder.append(")");
        // 表达式
        String expression = expressionBuilder.toString();
        LOCAL_LOG.info("表达式拼接结果: {}", expression);

        // 获取选项对应的名称
        ConditionOptionResponseVO responseVo = this.getOptions(conditionInfo.getId()).getData();
        if (ObjectUtil.isNull(responseVo) || CollectionUtil.isEmpty(responseVo.getOptionList())) {
            throw new RuntimeException("找不到条件的选项");
        }
        StringBuilder descriptionBuilder = new StringBuilder(conditionInfo.getConditionName() + ": ");
        for (CommonOptionResponseVO commonOptionResponseVo : responseVo.getOptionList()) {
            // 如果选中了这个选项，则拼接其name
            if (multipleOptions.contains(commonOptionResponseVo.getKey())) {
                descriptionBuilder.append(commonOptionResponseVo.getName()).append("/");
            }
        }
        String description = descriptionBuilder.toString();
        LOCAL_LOG.info("条件描述: {}", description);

        match.setOperatorExpression(expression);
        match.setOperatorDescription(description);
    }

    /**
     * 处理分组求并类型
     * @param match 匹配关系
     * @param conditionDto 请求条件
     * @param conditionInfo 条件信息
     */
    private void handleGroupOr(CrowdPackageConditionMatch match, CrowdPackageConditionDTO conditionDto, CrowdPackageCondition conditionInfo) {
        // 分组求并条件
        LOCAL_LOG.info("是分组求并条件");
        List<String> multipleOptions = conditionDto.getMultipleOptions();
        if (CollectionUtil.isEmpty(multipleOptions)) {
            return;
        }
        // 获取该条件的所有的条件
        ConditionOptionResponseVO responseVo = this.getOptions(conditionInfo.getId()).getData();
        if (ObjectUtil.isNull(responseVo) || CollectionUtil.isEmpty(responseVo.getOptionList())) {
            throw new RuntimeException("找不到条件的选项");
        }
        // 构造条件描述
        StringBuilder descriptionBuilder = new StringBuilder(conditionInfo.getConditionName() + "取档: ");
        for (CommonOptionResponseVO commonOptionResponseVo : responseVo.getOptionList()) {
            // 如果选中了这个选项，则拼接其name
            if (multipleOptions.contains(commonOptionResponseVo.getKey())) {
                descriptionBuilder.append(commonOptionResponseVo.getName()).append("/");
            }
        }
        String description = descriptionBuilder.toString();
        LOCAL_LOG.info("条件描述: {}", description);

        StringBuilder expressionBuilder = new StringBuilder("(");
        LOCAL_LOG.info("拼接表达式");
        int location = 1;
        for (String option : multipleOptions) {
            boolean isLast = location == multipleOptions.size();
            expressionBuilder.append("(").append(option).append(")");
            if (!isLast) {
                // 给非最后一个元素后面拼接 or
                expressionBuilder.append(" or ");
            }
            location += 1;
        }
        expressionBuilder.append(")");
        String expression = expressionBuilder.toString();
        LOCAL_LOG.info("表达式拼接结果: {}", expression);

        match.setOperatorDescription(description);
        match.setOperatorExpression(expression);

    }



}
