package com.yaoyaozw.customer.components;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.yaoyaozw.customer.common.R;
import com.yaoyaozw.customer.constants.ApiResultConstant;
import com.yaoyaozw.customer.constants.CustomerCommonConstant;
import com.yaoyaozw.customer.constants.RabbitCommonNameConstant;
import com.yaoyaozw.customer.entity.*;
import com.yaoyaozw.customer.enums.CustomerStoreTemplateEnum;
import com.yaoyaozw.customer.feigns.ReferralFeignClient;
import com.yaoyaozw.customer.mapper.CustomerGraphicsDelayMapper;
import com.yaoyaozw.customer.mapper.CustomerGraphicsMapper;
import com.yaoyaozw.customer.mapper.KanbanCommonMapper;
import com.yaoyaozw.customer.service.*;
import com.yaoyaozw.customer.utils.TencentCustomerUtil;
import com.yaoyaozw.customer.vo.AuthInfoVO;
import com.yaoyaozw.customer.vo.TencentMediaResponseVO;
import com.yaoyaozw.customer.vo.kanban.CommonOptionResponseVO;
import com.yaoyaozw.customer.vo.customer.CrowdPackageUserVO;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

import static com.yaoyaozw.customer.constants.CustomerCommonConstant.CUSTOMER_TYPE_VALUE_MINI_PROGRAM;
import static com.yaoyaozw.customer.constants.CustomerCommonConstant.*;

/**
 * @author darker
 * @date 2022/9/28 15:16
 */
@Component
public class CustomerServiceCommonAsyncComponent {

    private final static Logger LOCAL_LOG = LoggerFactory.getLogger(CustomerServiceCommonAsyncComponent.class);

    private final static String OVER_FREQUENCY_STR = "over_frequency";
    private final static String OVER_FREQUENCY_MSG = "访问速度过快";

    @Autowired
    private RegisterUserEntityService userEntityService;
    @Autowired
    private ReferralEntityService referralEntityService;
    @Autowired
    private CrowdPackageConditionMatchService conditionMatchService;
    @Autowired
    private SnowflakeComponent snowflakeComponent;
    @Autowired
    private CrowdPackageConditionMatchService matchService;
    @Autowired
    private RedisTemplate<String, Object> redisTemplate;
    @Autowired
    private ReferralFeignClient referralFeignClient;
    @Autowired
    private CustomerServiceCommonService customerServiceCommonService;
    @Autowired
    private CustomerGraphicsMapper customerGraphicsMapper;
    @Autowired
    private KanbanCommonMapper kanbanCommonMapper;
    @Autowired
    private CustomerGraphicsDelayMapper customerGraphicsDelayMapper;
    @Autowired
    private RabbitTemplate rabbitTemplate;
    @Resource
    private CompanyAcquisitionLinkService companyAcquisitionLinkService;
    @Autowired
    private TencentCustomerUtil tencentCustomerUtil;


    @Async("myExecutor")
    public void addMatchUserIntoPackage(Long packageId, Boolean needRemove) {
        // 获取符合人群包条件的用户列表
        List<CrowdPackageUserVO> userList = matchService.getUserListFromPackage(packageId, null);

        redisTemplate.opsForHash().put(CustomerCommonConstant.CROWD_HUMAN_NUN_REDIS_KEY, packageId.toString(), userList.size());
        long startTime = System.currentTimeMillis();
        String packIdStr = packageId.toString();
        // 筛选当前不在这个人群包的用户
        List<CrowdPackageUserVO> userNotInPackageList = userList.stream()
                // 保留现在不在该人群包的用户
                .filter(item -> StringUtils.isBlank(item.getInPackage()) || !item.getInPackage().contains(packIdStr))
                .peek(item -> {
                    if (StringUtils.isBlank(item.getInPackage())) {
                        item.setInPackage(packIdStr);
                    } else {
                        // 将当前人群包拼在后面
                        item.setInPackage(item.getInPackage() + "," + packIdStr);
                    }
                }).collect(Collectors.toList());

        LOCAL_LOG.info("人群包ID: {} 新增用户 {}个", packIdStr, userNotInPackageList.size());
        if (needRemove) {
            // 当前在人群包内的用户
            List<CrowdPackageUserVO> packCurrentUserList = userEntityService.getCurrentInPackUserList(packageId,true);
            if (CollectionUtil.isNotEmpty(packCurrentUserList)) {
                List<String> userOpenIdList = userList.stream().map(CrowdPackageUserVO::getOpenId).collect(Collectors.toList());
                // 筛选出之前在人群包内，现在不在地用户列表
                List<CrowdPackageUserVO> needRemoveUserList = packCurrentUserList.stream().filter(item -> !userOpenIdList.contains(item.getOpenId())).collect(Collectors.toList());
                removeUnMatchUserFromPackage(packageId, needRemoveUserList);
            }
        }
        if (CollectionUtil.isNotEmpty(userNotInPackageList)) {
            userEntityService.updateBatchById(encapsulateUserEntity(userNotInPackageList));
        }

        long endTime = System.currentTimeMillis();
        LOCAL_LOG.info("异步添加符合条件用户人群包完成, 耗时: {}ms", endTime - startTime);
    }

    @Async("myExecutor")
    public void removeUnMatchUserFromPackage(Long packageId, List<CrowdPackageUserVO> userList) {
        long startTime = System.currentTimeMillis();
        String packIdStr = packageId.toString();
        // 筛选当前不在这个人群包的用户
        List<CrowdPackageUserVO> userRemoveFromPackList = userList.stream()
                // 移除现在在该人群包但不符合当前条件的用户
                .filter(item -> StringUtils.isNotBlank(item.getInPackage()) && item.getInPackage().contains(packIdStr))
                .peek(item -> {
                    String removePackageStr = item.getInPackage().replace("," + packIdStr, "")
                            .replace(packIdStr + ",", "")
                            .replace(packIdStr, "");
                    item.setInPackage(StringUtils.isBlank(removePackageStr) ? "" : removePackageStr);
                }).collect(Collectors.toList());

        LOCAL_LOG.info("人群包ID: {} 移除用户 {}个", packIdStr, userRemoveFromPackList.size());
        if (CollectionUtil.isNotEmpty(userRemoveFromPackList)) {
            userEntityService.updateBatchById(encapsulateUserEntity(userRemoveFromPackList));
        }

        long endTime = System.currentTimeMillis();
        LOCAL_LOG.info("异步删除不符合条件用户所属人群包完成, 耗时: {}ms", endTime - startTime);
    }

    @Async("myExecutor")
    public void obtainMessageLink(Long materialId, CustomerGraphics customerGraphics, List<ReferralEntity> referralEntityList) {
        // 获取符合人群包条件的用户所属的公众号列表
        List<CrowdPackageUserVO> userListFromPackage = conditionMatchService.getUserListFromPackage(customerGraphics.getPackId(), null);
        Map<String, List<CrowdPackageUserVO>> storeGroupMap = userListFromPackage.stream().collect(Collectors.groupingBy(CrowdPackageUserVO::getStoreType));
        LOCAL_LOG.info("当前包含书城: {} 个", storeGroupMap.size());

        // 删除之前设置人群包的时候获取链接生成的数据
        referralEntityService.ultimateDeleteReferrals(materialId, "and account_id is not null");

        for (ReferralEntity referralEntity : referralEntityList) {
            // 不管有没有，先赋值图片地址
            referralEntity.setMediaOriginUrl(customerGraphics.getCoverUrl());
            referralEntity.setCustomerMsgType(customerGraphics.getType());
            // 获取链接
            if (CustomerCommonConstant.REMOTE_LINK_NEWS_TYPE_LIST.contains(referralEntity.getNewsType()) || 
                CustomerCommonConstant.ACQUISITION_LINK_NEWS_TYPE.equals(referralEntity.getNewsType())) {
                getMessageAuthListLink(storeGroupMap, customerGraphics, referralEntity);
            } else {
                LOCAL_LOG.info("newsType: {}, 不需要获取链接", referralEntity.getNewsType());
            }
        }
        // 处理完之后将状态改为待发送
        customerGraphics.setSendStatus(CustomerCommonConstant.SEND_STATUS_ACTIVE);
        customerGraphicsMapper.updateById(customerGraphics);
    }

    @Async("myExecutor")
    public void obtainDelayGraphicsLink(List<CustomerGraphicsDelay> sourceCustomerList, AuthorizerInfo targetAuthInfo) {

        String accountName = targetAuthInfo.getNickName();
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd");
        String dateStr = dateFormat.format(new Date());
        LOCAL_LOG.info("延时客服复用--公众号【{}】, 要复制客服数量: {} 条", accountName, sourceCustomerList.size());

        // 遍历客服素材进行处理
        for (CustomerGraphicsDelay sourceGraphics : sourceCustomerList) {
            dispatchCustomerDelayLink(dateStr, sourceGraphics, targetAuthInfo);
        }
    }

    public void dispatchDelay(String dateStr, AuthInfoVO authInfoVo, ReferralEntity referralEntity) {
        String storeType = referralEntity.getStoreType();
        if (storeType.equals(CustomerCommonConstant.STORE_NAME_YANG_GUANG) || storeType.equals(CustomerCommonConstant.STORE_NAME_ZHANG_ZHONG_YUN)) {
            if (CustomerCommonConstant.ACTIVITY_NEWS_TYPE.equals(referralEntity.getNewsType())) {
                // 获取config
                R activityConfig = referralFeignClient.getActivityConfig(referralEntity.getRechargeAmount().stripTrailingZeros().toPlainString(), storeType, null);
                // 获取资源配置列表
                Map<String, Object> data = activityConfig.getData();

                JSONArray configList = JSONUtil.parseArray(data.get("configList"));
                List<CommonOptionResponseVO> activityConfigList = JSONUtil.toList(configList, CommonOptionResponseVO.class);

                if (!CollectionUtil.isEmpty(activityConfigList)) {
                    // 转换config
                    String activityContent = referralEntity.getRechargeAmount().stripTrailingZeros().toPlainString() + "送" + referralEntity.getGiftAmount();
                    Map<String, String> configMap = activityConfigList.stream().collect(Collectors.toMap(key -> key.getLabel().replaceAll("[^(0-9.送)]", ""), CommonOptionResponseVO::getValue, (o1, o2) -> o2));
                    String configId = configMap.get(activityContent);
                    LOCAL_LOG.info("活动内容: {} 获取到configId: {}", activityContent, configId);
                    referralEntity.setConfigId(configId);
                    if (ObjectUtil.isNull(configId)) {
                        LOCAL_LOG.info("无法获取资源配置");
                    }
                } else {
                    LOCAL_LOG.info("无法获取资源配置");
                }
            }
        }
        getCopyReferral(dateStr, authInfoVo, referralEntity);

    }


    /**
     * 以下是客服消息模块的私有方法
     */
    private void getMessageAuthListLink(Map<String, List<CrowdPackageUserVO>> storeGroupMap, CustomerGraphics customerMain, ReferralEntity customerReferral) {
        Long packId = customerMain.getPackId();
        LOCAL_LOG.info("开始异步处理客服链接, 客服标题: {}, 人群包主键: {}", customerMain.getName(), packId);

        // 遍历处理书城
        Integer newsType = customerReferral.getNewsType();
        // 一个书城一个书城地处理
        if (newsType.equals(CustomerCommonConstant.BOOK_NEWS_TYPE)) {
            // 是推广链接
            generateMessageExtendLink(storeGroupMap, customerReferral);
        } else if (CustomerCommonConstant.ACTIVITY_NEWS_TYPE.equals(newsType)) {
            // 是活动链接
            generateMessageActivityLink(storeGroupMap, customerReferral);
        } else if (CustomerCommonConstant.USUAL_LINK_NEWS_TYPE.equals(newsType)) {
            // 常用链接
            generateMessageUsualLink(storeGroupMap, customerReferral);
        } else if (CustomerCommonConstant.ACQUISITION_LINK_NEWS_TYPE.equals(newsType)) {
            // 获客链接
            generateMessageAcquisitionLink(storeGroupMap, customerReferral);
        }

    }

    private void generateMessageExtendLink(Map<String, List<CrowdPackageUserVO>> storeGroupMap, ReferralEntity customerReferral) {

        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd");
        String dateStr = dateFormat.format(new Date());
        Map<String, String> storeEntityMap = getStoreEntityMap(storeGroupMap.keySet());

        storeGroupMap.forEach((storeType, userVoList) -> {
            LOCAL_LOG.info("当前处理书城: {}", storeType);
            // 去重提取公众号
            Set<String> accountSet = userVoList.stream().map(CrowdPackageUserVO::getAccountId).collect(Collectors.toSet());
            List<AuthInfoVO> authInfoList = customerServiceCommonService.getAuthInfoList(accountSet);
            // 定义
            List<ReferralEntity> referralEntityList = new ArrayList<>(accountSet.size());
            for (AuthInfoVO authInfoVo : authInfoList) {
                ReferralEntity referralEntity = new ReferralEntity();

                BeanUtil.copyProperties(customerReferral, referralEntity);
                // 重新生成链接主键id
                referralEntity.setId(snowflakeComponent.snowflakeId());
                referralEntity.setAccountId(authInfoVo.getAccountId());
                referralEntity.setStoreTypeName(storeEntityMap.get(storeType));
                try {
                    getCopyReferral(dateStr, authInfoVo, referralEntity);
                } catch (Exception e) {
                    LOCAL_LOG.warn("公众号: {} 获取链接异常: {}", authInfoVo.getAccountName(), e.getMessage());
                }
                // 所有处理结束之后，重新赋值 referralEntity 的accountId 为公众号本身的Id（主要是因为客服消息生成之后，发送的时候，referral_entity 表只有 account_id 字段可以区分推送公众号）进行保存
                referralEntity.setAccountId(authInfoVo.getAccountId());
                referralEntityList.add(referralEntity);
            }

            // 批量新增
            referralEntityService.saveBatch(referralEntityList);
        });
    }

    private void generateMessageActivityLink(Map<String, List<CrowdPackageUserVO>> storeGroupMap, ReferralEntity customerReferral) {

        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd");
        String dateStr = dateFormat.format(new Date());
        String activityContent = customerReferral.getRechargeAmount().stripTrailingZeros().toPlainString() + "送" + customerReferral.getGiftAmount();
        LOCAL_LOG.info("活动内容: {}", activityContent);
        Map<String, String> storeEntityMap = getStoreEntityMap(storeGroupMap.keySet());

        storeGroupMap.forEach((storeType, userVoList) -> {
            LOCAL_LOG.info("当前处理书城: {}", storeType);
            boolean isLegal = true;
            String configId = null;
            // 阳光、掌中云 需要获取configId
            if (storeType.equals(CustomerCommonConstant.STORE_NAME_YANG_GUANG) || storeType.equals(CustomerCommonConstant.STORE_NAME_ZHANG_ZHONG_YUN)) {
                R activityConfig = referralFeignClient.getActivityConfig(customerReferral.getRechargeAmount().stripTrailingZeros().toPlainString(), storeType, null);
                // 获取资源配置列表
                Map<String, Object> data = activityConfig.getData();

                JSONArray configList = JSONUtil.parseArray(data.get("configList"));
                List<CommonOptionResponseVO> activityConfigList = JSONUtil.toList(configList, CommonOptionResponseVO.class);

                if (!CollectionUtil.isEmpty(activityConfigList)) {
                    // 转换config
                    Map<String, String> configMap = activityConfigList.stream().collect(Collectors.toMap(key -> key.getLabel().replaceAll("[^(0-9.送)]", ""), CommonOptionResponseVO::getValue, (o1, o2) -> o2));
                    configId = configMap.get(activityContent);
                    LOCAL_LOG.info("活动内容: {} 获取到configId: {}", activityContent, configId);
                    if (ObjectUtil.isNull(configId)) {
                        isLegal = false;
                    }
                } else {
                    isLegal = false;
                    LOCAL_LOG.info("无法获取资源配置, 中断当前循环");
                }
            }
            customerReferral.setConfigId(configId);

            String tempId = CustomerStoreTemplateEnum.getTempId(storeType);
            customerReferral.setTemplateId(tempId);

            if (isLegal) {
                // 去重提取公众号
                Set<String> accountSet = userVoList.stream().map(CrowdPackageUserVO::getAccountId).collect(Collectors.toSet());
                List<AuthInfoVO> authInfoList = customerServiceCommonService.getAuthInfoList(accountSet);

                // 定义
                List<ReferralEntity> referralEntityList = new ArrayList<>(accountSet.size());
                for (AuthInfoVO authInfoVo : authInfoList) {
                    ReferralEntity referralEntity = new ReferralEntity();

                    BeanUtil.copyProperties(customerReferral, referralEntity);
                    // 重新生成链接主键id
                    referralEntity.setId(snowflakeComponent.snowflakeId());
                    referralEntity.setAccountId(authInfoVo.getAccountId());
                    referralEntity.setStoreTypeName(storeEntityMap.get(storeType));
                    try {
                        getCopyReferral(dateStr, authInfoVo, referralEntity);
                    } catch (Exception e) {
                        LOCAL_LOG.warn("公众号: {} 获取链接异常: {}", authInfoVo.getAccountName(), e.getMessage());
                    }
                    // 所有处理结束之后，重新赋值 referralEntity 的accountId 为公众号本身的Id（主要是因为客服消息生成之后，发送的时候，referral_entity 表只有 account_id 字段可以区分推送公众号）进行保存
                    referralEntity.setAccountId(authInfoVo.getAccountId());
                    referralEntityList.add(referralEntity);
                }

                // 批量新增
                LOCAL_LOG.info("保存书城: {} 的链接实体, 数量:{}条", storeType, referralEntityList.size());
                referralEntityService.saveBatch(referralEntityList);
            }
        });
    }

    private void generateMessageUsualLink(Map<String, List<CrowdPackageUserVO>> storeGroupMap, ReferralEntity customerReferral) {
        Map<String, String> storeEntityMap = getStoreEntityMap(storeGroupMap.keySet());
        // 获取常用链接映射关系
        Map<String, Map<String, String>> storeToCurrentToOriginMap = customerServiceCommonService.storeCommonLinkMap(customerReferral.getName());

        storeGroupMap.forEach((storeType, userVoList) -> {
            LOCAL_LOG.info("当前处理书城: {}", storeType);
            // 去重提取公众号
            Set<String> accountSet = userVoList.stream().map(CrowdPackageUserVO::getAccountId).collect(Collectors.toSet());
            List<AuthInfoVO> authInfoList = customerServiceCommonService.getAuthInfoList(accountSet);

            Map<String, String> currentToOriginMap = storeToCurrentToOriginMap.get(storeType.replaceAll("_[0-9]", ""));

            // 定义
            List<ReferralEntity> referralEntityList = new ArrayList<>(accountSet.size());
            for (AuthInfoVO authInfoVo : authInfoList) {
                ReferralEntity referralEntity = new ReferralEntity();

                BeanUtil.copyProperties(customerReferral, referralEntity);
                // 重新生成链接主键id
                referralEntity.setId(snowflakeComponent.snowflakeId());
                referralEntity.setAccountId(authInfoVo.getAccountId());
                referralEntity.setStoreTypeName(storeEntityMap.get(storeType));

                String name = referralEntity.getName();
                LOCAL_LOG.info("获取到常用链接当前key: {}", name);
                String originKey = currentToOriginMap.get(name);
                LOCAL_LOG.info("获取到原始常用链接key: {}", originKey);
                referralEntity.setName(originKey);

                // 常用链接不需要日期
                try {
                    getCopyReferral(null, authInfoVo, referralEntity);
                } catch (Exception e) {
                    LOCAL_LOG.warn("公众号: {} 获取链接异常: {}", authInfoVo.getAccountName(), e.getMessage());
                }
                // 所有处理结束之后，重新赋值 referralEntity 的accountId 为公众号本身的Id（主要是因为客服消息生成之后，发送的时候，referral_entity 表只有 account_id 字段可以区分推送公众号）进行保存
                referralEntity.setAccountId(authInfoVo.getAccountId());
                referralEntityList.add(referralEntity);
            }

            // 批量新增
            referralEntityService.saveBatch(referralEntityList);
        });
    }

    private void generateMessageAcquisitionLink(Map<String, List<CrowdPackageUserVO>> storeGroupMap, ReferralEntity customerReferral) {
        LOCAL_LOG.info("开始处理获客链接");
        Map<String, String> storeEntityMap = getStoreEntityMap(storeGroupMap.keySet());
        
        // 缓存公众号appId对应的获客链接，避免重复查询数据库
        Map<String, String> appIdAcquisitionLinkMap = new HashMap<>();

        storeGroupMap.forEach((storeType, userVoList) -> {
            LOCAL_LOG.info("当前处理书城: {}", storeType);
            // 去重提取公众号
            Set<String> accountSet = userVoList.stream().map(CrowdPackageUserVO::getAccountId).collect(Collectors.toSet());
            List<AuthInfoVO> authInfoList = customerServiceCommonService.getAuthInfoList(accountSet);

            // 定义
            List<ReferralEntity> referralEntityList = new ArrayList<>(accountSet.size());
            for (AuthInfoVO authInfoVo : authInfoList) {
                ReferralEntity referralEntity = new ReferralEntity();

                BeanUtil.copyProperties(customerReferral, referralEntity);
                // 重新生成链接主键id
                referralEntity.setId(snowflakeComponent.snowflakeId());
                referralEntity.setAccountId(authInfoVo.getAccountId());
                referralEntity.setStoreTypeName(storeEntityMap.get(storeType));

                try {
                    // 获取获客链接
                    String acquisitionLink = getAcquisitionLinkFromCache(authInfoVo.getAppId(), appIdAcquisitionLinkMap);
                    if (StringUtils.isNotBlank(acquisitionLink)) {
                        referralEntity.setReferral(acquisitionLink);
                        LOCAL_LOG.info("公众号: {} 获取到获客链接: {}", authInfoVo.getAccountName(), acquisitionLink);
                    } else {
                        LOCAL_LOG.warn("公众号: {} 未找到可用的获客链接，设置为error", authInfoVo.getAccountName());
                        referralEntity.setReferral(CustomerCommonConstant.ERROR_LABEL);
                    }
                } catch (Exception e) {
                    LOCAL_LOG.warn("公众号: {} 获取获客链接异常: {}", authInfoVo.getAccountName(), e.getMessage());
                    referralEntity.setReferral(CustomerCommonConstant.ERROR_LABEL);
                }

                referralEntityList.add(referralEntity);
            }

            // 批量新增
            LOCAL_LOG.info("保存书城: {} 的获客链接实体, 数量:{}条", storeType, referralEntityList.size());
            referralEntityService.saveBatch(referralEntityList);
        });
    }

    /**
     * 从缓存中获取获客链接，避免重复查询数据库
     * @param appId 公众号appId
     * @param appIdAcquisitionLinkMap 缓存Map
     * @return 获客链接，如果没有则返回null
     */
    private String getAcquisitionLinkFromCache(String appId, Map<String, String> appIdAcquisitionLinkMap) {
        // 先从缓存中查找
        if (appIdAcquisitionLinkMap.containsKey(appId)) {
            return appIdAcquisitionLinkMap.get(appId);
        }

        // 缓存中没有，查询数据库
        try {
            List<String> acquisitionLinkList = companyAcquisitionLinkService.getAcquisitionLinksByAppId(appId, "OPERATE");
            
            String acquisitionLink = null;
            if (CollectionUtil.isNotEmpty(acquisitionLinkList)) {
                // 取第一个可用的获客链接
                acquisitionLink = acquisitionLinkList.get(0);
                LOCAL_LOG.info("公众号: {} 查询到 {} 个可用获客链接，使用第一个: {}", appId, acquisitionLinkList.size(), acquisitionLink);
            } else {
                LOCAL_LOG.warn("公众号: {} 没有找到可用的获客链接", appId);
            }

            // 存入缓存（包括null值，避免重复查询）
            appIdAcquisitionLinkMap.put(appId, acquisitionLink);
            return acquisitionLink;
        } catch (Exception e) {
            LOCAL_LOG.error("查询公众号: {} 的获客链接时发生异常: {}", appId, e.getMessage(), e);
            // 异常情况也存入缓存，避免重复查询
            appIdAcquisitionLinkMap.put(appId, null);
            return null;
        }
    }

    /**
     * 以下是延时客服 的方法
     */

    private void dispatchCustomerDelayLink(String dateStr, CustomerGraphicsDelay sourceGraphics, AuthorizerInfo targetAuthInfo) {
        // 新建新主体
        CustomerGraphicsDelay targetCustomerGraphics = new CustomerGraphicsDelay();
        List<ReferralEntity> targetReferralEntityList = new ArrayList<>();

        BeanUtil.copyProperties(sourceGraphics, targetCustomerGraphics, "sourceUrl");
        long mainId = snowflakeComponent.snowflakeId();
        targetCustomerGraphics.setId(mainId);
        targetCustomerGraphics.setAuthorizerInfo(targetAuthInfo);
        targetCustomerGraphics.setAppId(targetAuthInfo.getAppid());
        List<ReferralEntity> sourceReferralList = sourceGraphics.getBelongReferralList();

        String type = sourceGraphics.getType();
        // 构造h5
        StringBuilder h5Context = new StringBuilder();

        // 遍历处理客服链接
        int idx = 1;
        boolean normal = true;
        boolean needLink = true;
        for (ReferralEntity sourceReferralEntity : sourceReferralList) {
            // 生成新的链接数据
            ReferralEntity targetReferralEntity = new ReferralEntity();
            BeanUtil.copyProperties(sourceReferralEntity, targetReferralEntity, "referral");
            if (sourceReferralEntity.getNewsType().equals(-1)) {
                // 自定义链接延用链接
                targetReferralEntity.setReferral(companyAcquisitionLinkService.checkAndSearchTargetReferral(sourceReferralEntity.getReferral(), targetAuthInfo.getId()));
            }
            // 重新生成链接主键id
            targetReferralEntity.setId(snowflakeComponent.snowflakeId());
            targetReferralEntity.setMaterialGraphicsId(mainId);
            targetReferralEntity.setAccountId(targetAuthInfo.getAccountId());
            targetReferralEntity.setStoreTypeName(targetAuthInfo.getStoreTypeName());

            Integer newsType = sourceReferralEntity.getNewsType();
            // 调用获取链接方法
            if (newsType.equals(CustomerCommonConstant.BOOK_NEWS_TYPE)) {
                // 是推广链接
                generateDelayExtendLink(dateStr, targetReferralEntity, targetAuthInfo);
            } else if (newsType.equals(CustomerCommonConstant.ACTIVITY_NEWS_TYPE)) {
                // 是活动链接
                generateDelayActivityLink(dateStr, targetReferralEntity, targetAuthInfo);
            } else if (newsType.equals(CustomerCommonConstant.USUAL_LINK_NEWS_TYPE)) {
                // 常用链接
                generateDelayUsualLink(dateStr, targetReferralEntity, targetAuthInfo);
            } else {
                needLink = false;
                LOCAL_LOG.info("newsType: {}, 不需要获取链接", newsType);
            }
            targetReferralEntityList.add(targetReferralEntity);

            // 图文主表只有一个素材链接，文本主表不需要存素材链接
            String referral = targetReferralEntity.getReferral();
            if (needLink && StringUtils.isBlank(referral)) {
                normal = false;
            }
            targetCustomerGraphics.setSourceUrl(referral);
            if (type.equals(CustomerCommonConstant.CUSTOMER_TYPE_VALUE_TEXT)) {
                // 文本类型替换h5链接
                String context = null;
                if (CustomerCommonConstant.REPLACE_LINK_NEWS_TYPE_LIST.contains(newsType)) {
                    context = CustomerCommonConstant.CUSTOMER_TEXT_LINK_TEMPLATE.replace(CustomerCommonConstant.CUSTOMER_TEXT_CONTENT_PLACEHOLDER, targetReferralEntity.getTextContent());
                } else if (CustomerCommonConstant.COMMON_NEWS_TYPE_LIST.contains(newsType)){
                    context = targetReferralEntity.getTextContent();
                }
                if (ObjectUtil.isNotNull(context)) {
                    // 拼接的要替换链接
                    if (StringUtils.isNotEmpty(referral)) {
                        h5Context.append(context.replace(CustomerCommonConstant.CUSTOMER_TEXT_URL_PLACEHOLDER, referral));
                    }else {
                        h5Context.append(context);
                    }
                    if (idx != sourceReferralList.size()) {
                        h5Context.append("\n").append("\n");
                    }
                }

                idx += 1;
            }
        }
        targetCustomerGraphics.setSendStatus(normal ? CustomerCommonConstant.SEND_STATUS_ACTIVE : CustomerCommonConstant.SEND_STATUS_LINK_ERROR);

        if (StringUtils.isNotBlank(h5Context)) {
            targetCustomerGraphics.setContent(h5Context.toString().replace(CustomerCommonConstant.H5_STYLE_CODE, ""));
        }
        LOCAL_LOG.info("保存主体数据, 主体id: {}", mainId);
        customerGraphicsDelayMapper.insert(targetCustomerGraphics);

        LOCAL_LOG.info("准备保存, 共生成链接数据 : {} 条", targetReferralEntityList.size());
        referralEntityService.saveBatch(targetReferralEntityList);
    }

    private void generateDelayExtendLink(String dateStr, ReferralEntity targetReferralEntity, AuthorizerInfo targetAuthInfo) {
        // 遍历链接数据进行处理
        try {
            // 获取链接
            AuthInfoVO authInfoVo = new AuthInfoVO();
            authInfoVo.putPropertyValue(targetAuthInfo);
            getCopyReferral(dateStr, authInfoVo, targetReferralEntity);
        } catch (Exception e) {
            LOCAL_LOG.warn("公众号: {} 获取链接异常: {}", targetAuthInfo.getNickName(), e.getMessage());
        }
    }

    private void generateDelayActivityLink(String dateStr, ReferralEntity targetReferralEntity, AuthorizerInfo targetAuthInfo) {
        String storeType = targetReferralEntity.getStoreType();
        boolean isLegal = true;
        String configId = null;
        String activityContent = targetReferralEntity.getRechargeAmount().stripTrailingZeros().toPlainString() + "送" + targetReferralEntity.getGiftAmount();
        // 阳光、掌中云 需要获取configId
        if (storeType.equals(CustomerCommonConstant.STORE_NAME_YANG_GUANG) || storeType.equals(CustomerCommonConstant.STORE_NAME_ZHANG_ZHONG_YUN)) {
            R activityConfig = referralFeignClient.getActivityConfig(targetReferralEntity.getRechargeAmount().stripTrailingZeros().toPlainString(), storeType, null);
            // 获取资源配置列表
            Map<String, Object> data = activityConfig.getData();

            JSONArray configList = JSONUtil.parseArray(data.get("configList"));
            List<CommonOptionResponseVO> activityConfigList = JSONUtil.toList(configList, CommonOptionResponseVO.class);

            if (!CollectionUtil.isEmpty(activityConfigList)) {
                // 转换config
                Map<String, String> configMap = activityConfigList.stream().collect(Collectors.toMap(key -> key.getLabel().replaceAll("[^(0-9.送)]", ""), CommonOptionResponseVO::getValue, (o1, o2) -> o2));
                configId = configMap.get(activityContent);
                LOCAL_LOG.info("活动内容: {} 获取到configId: {}", activityContent, configId);
                if (ObjectUtil.isNull(configId)) {
                    isLegal = false;
                }
            } else {
                isLegal = false;
                LOCAL_LOG.info("无法获取资源配置");
            }
        }
        targetReferralEntity.setConfigId(configId);
        String tempId = CustomerStoreTemplateEnum.getTempId(storeType);
        targetReferralEntity.setTemplateId(tempId);

        if (isLegal) {

            // 重新生成链接主键id
            targetReferralEntity.setId(snowflakeComponent.snowflakeId());
            targetReferralEntity.setAccountId(targetAuthInfo.getAccountId());
            try {
                // 获取链接
                AuthInfoVO authInfoVo = new AuthInfoVO();
                authInfoVo.putPropertyValue(targetAuthInfo);
                getCopyReferral(dateStr, authInfoVo, targetReferralEntity);
            } catch (Exception e) {
                LOCAL_LOG.warn("公众号: {} 获取链接异常: {}", targetAuthInfo.getNickName(), e.getMessage());
            }
        }


    }

    private void generateDelayUsualLink(String dateStr, ReferralEntity targetReferralEntity, AuthorizerInfo targetAuthInfo) {
        try {
            // 获取链接
            AuthInfoVO authInfoVo = new AuthInfoVO();
            authInfoVo.putPropertyValue(targetAuthInfo);
            getCopyReferral(dateStr, authInfoVo, targetReferralEntity);
        } catch (Exception e) {
            LOCAL_LOG.warn("公众号: {} 获取链接异常: {}", targetAuthInfo.getNickName(), e.getMessage());
        }
    }



    /**
     * 以下是公有方法
     */
    public void getCopyReferral(String dateStr, AuthInfoVO authInfoVo, ReferralEntity referralEntity) {
        if (CustomerCommonConstant.STORE_NAME_TOMATO.equals(authInfoVo.getStoreType())) {
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        ygAccessLimit(true, authInfoVo.getStoreType());
        referralEntity.setStoreType(authInfoVo.getStoreType());
        // 非常用链接类型的name需要处理
        Integer newsType = referralEntity.getNewsType();
        String name = CustomerCommonConstant.getLinkNameModel(newsType);
        if (StringUtils.isNotBlank(name) && !CustomerCommonConstant.USUAL_LINK_NEWS_TYPE.equals(newsType)) {
            name = name + "-" +System.currentTimeMillis() % 100000;
            referralEntity.setName(name);
        }
        if (!CustomerCommonConstant.USUAL_LINK_NEWS_TYPE.equals(newsType) && !StringUtils.isBlank(name)) {
            String newsTypeName = CustomerCommonConstant.getNewsTypeName(newsType);
            if (ObjectUtil.isNull(newsTypeName)) {
                throw new RuntimeException("无法获取链接类型");
            }
            // 替换共用字段
            if (StringUtils.isBlank(referralEntity.getStoreTypeName())) {
                referralEntity.setStoreTypeName("");
            }
            String bookName = referralEntity.getBookName();
            if (StringUtils.isNotBlank(bookName)) {
                bookName = bookName.substring(0, Math.min(8, bookName.length()));
            }
            name = name.replace("{newsType}", newsTypeName).replace("{accountNickName}", authInfoVo.getAccountName())
                    .replace("{storeType}", referralEntity.getStoreTypeName()).replace("{currentDate}", dateStr);
            if (newsType.equals(CustomerCommonConstant.BOOK_NEWS_TYPE)) {
                // 系统-客服-{newsType}-{accountNickName}-{storeType}-{currentDate}-{bookName}
                name = name.replace("{bookName}", bookName);
            } else if (newsType.equals(CustomerCommonConstant.ACTIVITY_NEWS_TYPE)) {
                // 系统-客服-{newsType}-{accountNickName}-{storeType}-{currentDate}-充{recharge}送{gift}
                name = name.replace("{recharge}", referralEntity.getRechargeAmount().stripTrailingZeros().toPlainString()).replace("{gift}", referralEntity.getGiftAmount().toString());
                String tempId = CustomerStoreTemplateEnum.getTempId(referralEntity.getStoreType());
                referralEntity.setTemplateId(tempId);
            }
            referralEntity.setName(name);
        }
        referralEntity.setInfoId(authInfoVo.getId());

        // 判断如果是番茄小程序的话，处理参数
        if (CustomerCommonConstant.STORE_NAME_TOMATO.equals(authInfoVo.getStoreType()) && CUSTOMER_TYPE_VALUE_MINI_PROGRAM.equals(referralEntity.getCustomerMsgType())) {
            // 番茄小程序
            if (CustomerCommonConstant.ACTIVITY_NEWS_TYPE.equals(referralEntity.getNewsType())) {
                // 活动使用公众号本身的分销商Id
                referralEntity.setAccountId(authInfoVo.getAccountId());
            } else {
                referralEntity.setAccountId(authInfoVo.getExpandDistributorId());
            }
        }
        // LOCAL_LOG.info("获取链接, 参数: {}", referralEntity);
        R r = referralFeignClient.productReferral(referralEntity);
        if (!r.getCode().equals(ApiResultConstant.SUCCESS_CODE)) {
            if (OVER_FREQUENCY_MSG.equals(r.getMessage())) {
                referralEntity.setReferral(OVER_FREQUENCY_STR);
            } else {
                referralEntity.setReferral(null);
            }
            LOCAL_LOG.info("公众号：{} 获取链接失败：{}, 参数: {}", authInfoVo.getAccountName(), r.getMessage(), referralEntity);
            throw new RuntimeException(r.getMessage());
        } else {
            LOCAL_LOG.info("公众号: {} 获取链接成功", authInfoVo.getAccountName());
        }
        String res = r.getData("storeReferral", new TypeReference<String>() {});
        JSONObject jsonObject1 = JSON.parseObject(res);
        String referral = jsonObject1.getString("referral");
        referralEntity.setPromoteId(jsonObject1.getString("promoteId"));
        referralEntity.setReferral(referral);
        // 取链接成功之后，后置处理番茄小程序
        suffixHandleTomatoMiniProgram(authInfoVo, referralEntity);
    }


    private void suffixHandleTomatoMiniProgram(AuthInfoVO authInfoVo, ReferralEntity referralEntity) {
        if (CustomerCommonConstant.STORE_NAME_TOMATO.equals(authInfoVo.getStoreType()) && CUSTOMER_TYPE_VALUE_MINI_PROGRAM.equals(referralEntity.getCustomerMsgType())) {
            String referral = referralEntity.getReferral();
            // 番茄小程序
            if (CustomerCommonConstant.ACTIVITY_NEWS_TYPE.equals(referralEntity.getNewsType())) {
                // 活动需要后置查询番茄小程序链接
                // 根据 promoteId 查询番茄链接详情，番茄活动接口创建链接返回的是http链接，需要查询接口获取其小程序 /page 链接
                R r = referralFeignClient.queryActivityInfo(referralEntity.getAccountId(), referralEntity.getPromoteId());
                if (!r.getCode().equals(ApiResultConstant.SUCCESS_CODE)) {
                    throw new RuntimeException("获取活动链接详情失败: " + r.getMessage());
                }
                String res = r.getData("activityData", new TypeReference<String>() {});
                JSONObject jsonObject = JSON.parseObject(res);
                referral = jsonObject.getString("url");
            }

            if (StringUtils.isNotBlank(referral) && isMiniProgramPath(referral)) {
                // 添加 get_book_item_id=true 参数
                if (!referral.contains(GET_BOOK_ITEM)) {
                    referral = referral + "&" + GET_BOOK_ITEM_VAL;
                }
                // 添加 from_oa_app_id={flagId} 参数
                if (!referral.contains(FROM_APPID)) {
                    referral = referral + "&" + FROM_APPID + "=" + authInfoVo.getExpandFlagId();
                }

                // 添加 landing_page=reader 参数
                if (!referral.contains(LANDING_PAGE) && !CustomerCommonConstant.USUAL_LINK_NEWS_TYPE.equals(referralEntity.getNewsType())) {
                    referral = referral + "&" + LANDING_PAGE_VAL;
                }
                // 常用链接换掉scene
                if (CustomerCommonConstant.USUAL_LINK_NEWS_TYPE.equals(referralEntity.getNewsType())) {
                    referral = referral.replace("scene=0", "scene=1");
                }
            }
            referralEntity.setMpAppId(authInfoVo.getExpandMpAppId());
            referralEntity.setMpPath(referral);
            // 暂时将小程序地址也赋值到跳转链接地址上
            referralEntity.setReferral(referral);


            // 上传腾讯图片素材
            TencentMediaResponseVO uploadResult = tencentCustomerUtil.uploadTencentMedia(authInfoVo.getAppId(), referralEntity.getMediaOriginUrl(), referralEntity.getCustomerMsgType());
            if (StringUtils.isNotEmpty(uploadResult.getErrmsg())) {
                LOCAL_LOG.warn("公众号: {} 上传素材失败 {}", authInfoVo.getAppId(), uploadResult.getErrmsg());
            }
            referralEntity.setMediaId(uploadResult.getMedia_id());
        }
    }
    private Boolean isMiniProgramPath(String path) {
        return StringUtils.isNotBlank(path) && (path.startsWith("page") || path.startsWith("/page"));
    }

    private Map<String, String> getStoreEntityMap(Set<String> storeKeySet) {
        if (CollectionUtil.isEmpty(storeKeySet)) {
            return new HashMap<>(4);
        }
        List<CommonOptionResponseVO> storeTypeEntity = kanbanCommonMapper.getStoreTypeEntity(storeKeySet);
        return storeTypeEntity.stream().collect(Collectors.toMap(CommonOptionResponseVO::getKey, CommonOptionResponseVO::getName));

    }

    private List<RegisterUserEntity> encapsulateUserEntity(List<CrowdPackageUserVO> userList) {
        List<RegisterUserEntity> userEntityList = new ArrayList<>();
        for (CrowdPackageUserVO crowdPackageUserVO : userList) {
            RegisterUserEntity registerUserEntity = new RegisterUserEntity();
            registerUserEntity.setId(crowdPackageUserVO.getId());
            registerUserEntity.setInPackage(crowdPackageUserVO.getInPackage());
            userEntityList.add(registerUserEntity);
        }
        return userEntityList;
    }

    private void ygAccessLimit(Boolean checkStoreType, String storeType) {
        if (checkStoreType && !CustomerCommonConstant.STORE_NAME_YANG_GUANG.equals(storeType)) {
            // 需要校验书城但是不是阳光书城，直接返回
            return;
        }
        boolean go;
        synchronized (rabbitTemplate) {
            // 从redis查询计数
            Object count = redisTemplate.opsForValue().get(CustomerCommonConstant.YANG_GUANG_ACCESS_LIMIT_REDIS_KEY);
            if (ObjectUtil.isNull(count)) {
                // redis中没有，设置0
                redisTemplate.opsForValue().set(CustomerCommonConstant.YANG_GUANG_ACCESS_LIMIT_REDIS_KEY, 0);
                count = 0;
            }

            LOCAL_LOG.info("当前count: {}", count);
            long stamp = System.currentTimeMillis();
            // 判断计数是否达到上限
            go = (Integer)count < 80;

            if (go) {
                // 没有达到上限，发送消息并且计数加1
                rabbitTemplate.convertAndSend(RabbitCommonNameConstant.OPERATE_COMMON_EXCHANGE, RabbitCommonNameConstant.YG_LIMIT_TTL_ROUTE_KEY, stamp);
                redisTemplate.opsForValue().increment(CustomerCommonConstant.YANG_GUANG_ACCESS_LIMIT_REDIS_KEY);
            }
        }
        if (!go) {
            // 计数达到上限了，休眠30秒，递归调用
            try {
                LOCAL_LOG.info("达到上限, 休眠15秒");
                Thread.sleep(15000);
                // 再次调用不需要校验书城
                ygAccessLimit(false, "");
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

    }




}
