package com.yaoyaozw.customer.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yaoyaozw.customer.constants.CrowdPackageCommonConstant;
import com.yaoyaozw.customer.entity.CrowdPackage;
import com.yaoyaozw.customer.entity.CrowdPackageConditionMatch;
import com.yaoyaozw.customer.entity.RegisterUserEntity;
import com.yaoyaozw.customer.mapper.KanbanCommonMapper;
import com.yaoyaozw.customer.mapper.MaterialCrowdConditionMatchMapper;
import com.yaoyaozw.customer.service.CrowdPackageConditionMatchService;
import com.yaoyaozw.customer.service.CrowdPackageService;
import com.yaoyaozw.customer.service.CustomerServiceCommonService;
import com.yaoyaozw.customer.service.RegisterUserEntityService;
import com.yaoyaozw.customer.vo.customer.CrowdPackageUserVO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author darker
 * @date 2022/9/21 15:09
 */
@Service
public class CrowdPackageConditionMatchServiceImpl extends ServiceImpl<MaterialCrowdConditionMatchMapper, CrowdPackageConditionMatch> implements CrowdPackageConditionMatchService {

    private final static Logger LOCAL_LOG = LoggerFactory.getLogger(CrowdPackageConditionMatchServiceImpl.class);

    @Autowired
    private CustomerServiceCommonService commonService;
    @Autowired
    private RegisterUserEntityService userEntityService;
    @Lazy
    @Autowired
    private CrowdPackageService crowdPackageService;

    @Override
    public List<CrowdPackageUserVO> getUserListFromPackage(Long packageId, String openId) {
        long startTime = System.currentTimeMillis();

        // 获取人群包下的条件
        List<CrowdPackageConditionMatch> packageConditionList = super.list(new QueryWrapper<CrowdPackageConditionMatch>().eq("package_id", packageId));
        List<String> staticConditionList = packageConditionList.stream().filter(item -> CrowdPackageCommonConstant.STATIC_CONDITION.equals(item.getIsStatic())).map(CrowdPackageConditionMatch::getOperatorExpression).collect(Collectors.toList());
        List<String> nonStaticConditionList = packageConditionList.stream().filter(item -> CrowdPackageCommonConstant.NON_STATIC_CONDITION.equals(item.getIsStatic())).map(CrowdPackageConditionMatch::getOperatorExpression).collect(Collectors.toList());
        LOCAL_LOG.info("静态条件: {}条, 动态条件: {}条", staticConditionList.size(), nonStaticConditionList.size());
        long getConditionTime = System.currentTimeMillis();
        LOCAL_LOG.info("获取条件列表耗时 {}ms", getConditionTime - startTime);

        // 根据静态条件获取setupId
        Set<Long> setupIdList = commonService.getSetupIdListFromStaticCondition(staticConditionList);
        long getSetupIdTime = System.currentTimeMillis();
        LOCAL_LOG.info("获取SetupId列表耗时 {}ms, setupId列表长度: {}", getSetupIdTime - getConditionTime, setupIdList.size());
        if (CollectionUtil.isEmpty(setupIdList)) {
            return new ArrayList<>();
        }

        // 根据动态条件获取用户列表
        List<CrowdPackageUserVO> userList = userEntityService.getUserMatchDynamicExpress(nonStaticConditionList, openId);
        long dynamicUserTime = System.currentTimeMillis();
        LOCAL_LOG.info("获取SetupId列表符合动态条件的用户耗时 {}ms, 符合动态条件的用户: {}个", dynamicUserTime - getSetupIdTime, userList.size());

        // 筛选所属setupId在列表中的用户
        userList = userList.stream().filter(item -> setupIdList.contains(item.getSetupId())).collect(Collectors.toList());

        long allUserTime = System.currentTimeMillis();
        LOCAL_LOG.info("获取符合全部条件的用户耗时: {}ms, 符合全部条件的用户: {}个", allUserTime - dynamicUserTime, userList.size());

        if (CollectionUtil.isEmpty(userList)) {
            LOCAL_LOG.info("人群包ID {}, 没有符合条件的用户", packageId);
        }
        // 获取人群包信息
        CrowdPackage crowdPackage = crowdPackageService.getById(packageId);
        if (crowdPackage == null) {
            LOCAL_LOG.warn("人群包ID {} 不存在", packageId);
            return new ArrayList<>();
        }
        // 根据关注时间范围过滤用户
        if (crowdPackage.getFollowDateStart() != null || crowdPackage.getFollowDateEnd() != null) {
            userList = userList.stream()
                    .filter(item -> isUserInFollowDateRange(item, crowdPackage.getFollowDateStart(), crowdPackage.getFollowDateEnd()))
                    .collect(Collectors.toList());
            LOCAL_LOG.info("人群包ID: {} 根据关注时间范围过滤后用户数量: {}", packageId, userList.size());
        }

        return userList;
    }

    @Override
    public Boolean getUserPackageBelong(Long packageId, String openId) {
        List<CrowdPackageUserVO> userList = this.getUserListFromPackage(packageId, openId);
        return CollectionUtil.isNotEmpty(userList);
    }


    /**
     * 判断用户是否在关注时间范围内
     * @param user 用户信息
     * @param followDateStart 关注开始日期
     * @param followDateEnd 关注结束日期
     * @return 是否在范围内
     */
    private boolean isUserInFollowDateRange(CrowdPackageUserVO user, Date followDateStart, Date followDateEnd) {
        Date userGmtCreate = user.getGmtCreate();
        if (userGmtCreate == null) {
            // 用户创建时间为空，不符合条件
            return false;
        }

        // 检查是否在开始日期之后（包含）
        if (followDateStart != null) {
            Date startDate = DateUtil.beginOfDay(followDateStart);
            if (userGmtCreate.before(startDate)) {
                return false;
            }
        }

        // 检查是否在结束日期之前（包含）
        if (followDateEnd != null) {
            Date endDate = DateUtil.endOfDay(followDateEnd);
            return !userGmtCreate.after(endDate);
        }

        return true;
    }

}
