package com.yaoyaozw.customer.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yaoyaozw.customer.common.GenericsResult;
import com.yaoyaozw.customer.components.CustomerServiceCommonAsyncComponent;
import com.yaoyaozw.customer.components.SnowflakeComponent;
import com.yaoyaozw.customer.components.TokenManager;
import com.yaoyaozw.customer.constants.CustomerCommonConstant;
import com.yaoyaozw.customer.dto.customer.CustomerDelayTextSaveDTO;
import com.yaoyaozw.customer.dto.customer.CustomerReferralDTO;
import com.yaoyaozw.customer.entity.AuthorizerInfo;
import com.yaoyaozw.customer.entity.CustomerGraphicsDelay;
import com.yaoyaozw.customer.entity.ReferralEntity;
import com.yaoyaozw.customer.mapper.CustomerGraphicsDelayMapper;
import com.yaoyaozw.customer.mapper.KanbanCommonMapper;
import com.yaoyaozw.customer.service.AuthorizerInfoService;
import com.yaoyaozw.customer.service.CustomerDelayTextService;
import com.yaoyaozw.customer.service.CustomerGraphicsDelayService;
import com.yaoyaozw.customer.service.ReferralEntityService;
import com.yaoyaozw.customer.vo.AuthInfoVO;
import com.yaoyaozw.customer.vo.customer.CustomerContentVO;
import com.yaoyaozw.customer.vo.customer.CustomerDelayTextDetailVO;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @author darker
 * @date 2022/10/17 16:39
 */
@Service
public class CustomerDelayTextServiceImpl extends ServiceImpl<CustomerGraphicsDelayMapper, CustomerGraphicsDelay> implements CustomerDelayTextService {

    private final static Logger localLog = LoggerFactory.getLogger(CustomerDelayTextServiceImpl.class);

    @Autowired
    private TokenManager tokenManager;
    @Autowired
    private SnowflakeComponent snowflakeComponent;
    @Autowired
    private ReferralEntityService referralEntityService;
    @Autowired
    private CustomerServiceCommonAsyncComponent commonAsyncComponent;
    @Autowired
    private KanbanCommonMapper kanbanCommonMapper;
    @Autowired
    private CustomerGraphicsDelayService graphicsDelayService;

    @Override
    public GenericsResult<String> insertCustomerDelay(CustomerDelayTextSaveDTO saveDto) {
        // 主体数据
        localLog.info("处理主体数据");
        CustomerGraphicsDelay customerGraphicsDelay = new CustomerGraphicsDelay();
        BeanUtil.copyProperties(saveDto, customerGraphicsDelay);
        boolean isCreate = ObjectUtil.isNull(customerGraphicsDelay.getId());
        customerGraphicsDelay.initOperateInfo(tokenManager.getUserIdFromToken(), ObjectUtil.isNull(saveDto.getId()));
        if (isCreate) {
            long id = snowflakeComponent.snowflakeId();
            customerGraphicsDelay.setId(id);
        }
        customerGraphicsDelay.setSendStatus(CustomerCommonConstant.SEND_STATUS_ACTIVE);
        super.saveOrUpdate(customerGraphicsDelay);

        String appId = saveDto.getAppId();
        if (StringUtils.isBlank(appId)) {
            // 更新的时候
            CustomerGraphicsDelay entity = super.getById(saveDto.getId());
            if (ObjectUtil.isNotNull(entity)) {
                appId = entity.getAppId();
            }
        }
        if (!StringUtils.isBlank(appId)) {
            graphicsDelayService.reSortCustomer(appId);
        }

        return new GenericsResult<>(customerGraphicsDelay.getId().toString());
    }

    @Override
    public GenericsResult<CustomerDelayTextDetailVO> insertCustomerContent(CustomerReferralDTO referralDto) {

        // 处理活动数据
        ReferralEntity referralEntity = new ReferralEntity();
        BeanUtil.copyProperties(referralDto, referralEntity);
        // 获取name模板
        String nameModel = CustomerCommonConstant.getLinkNameModel(referralEntity.getNewsType());
        localLog.info("获取name模板: {}", nameModel);
        if (StringUtils.isNotBlank(nameModel)) {
            referralEntity.setName(nameModel);
        }

        String storeTypeName = kanbanCommonMapper.getStoreTypeNameByAppId(referralDto.getAppId());
        referralEntity.setStoreTypeName(storeTypeName);
        // 获取链接
        // 日期
        String format = new SimpleDateFormat("yyyy/MM/dd").format(new Date());
        // 公众号
        AuthInfoVO authInfoVO = super.baseMapper.getCustomerDelayAuthInfo(referralDto.getMaterialGraphicsId());
        if (CustomerCommonConstant.REMOTE_LINK_NEWS_TYPE_LIST.contains(referralEntity.getNewsType())) {
            commonAsyncComponent.dispatchDelay(format, authInfoVO, referralEntity);
        }
        // 保存链接数据
        referralEntityService.saveOrUpdate(referralEntity);

        return getCustomerTextDetail(referralDto.getMaterialGraphicsId());
    }

    @Override
    public GenericsResult<CustomerDelayTextDetailVO> getCustomerTextDetail(Long id) {
        // 查询主体数据
        CustomerGraphicsDelay customerGraphicsDelay = super.getById(id);
        if (ObjectUtil.isNull(customerGraphicsDelay)) {
            return new GenericsResult<>(false, "找不到主体数据");
        }
        // 赋值主体数据
        CustomerDelayTextDetailVO detailVO = new CustomerDelayTextDetailVO();
        BeanUtil.copyProperties(customerGraphicsDelay, detailVO);

        // 查询所有的链接信息
        List<ReferralEntity> referralEntityList = referralEntityService.list(new QueryWrapper<ReferralEntity>().eq("material_graphics_id", id).orderByAsc("gmt_create").orderByAsc("sort"));
        List<CustomerContentVO> contentList = new ArrayList<>();
        StringBuilder mainH5Content = new StringBuilder();

        localLog.info("根据链接重新生成H5");
        // 位置计数器
        int idx = 1;
        for (ReferralEntity item : referralEntityList) {
            item.setSort(idx);
            CustomerContentVO customerContentVO = new CustomerContentVO();
            // 设置主键id和h5代码段
            customerContentVO.setId(item.getId());

            Integer newsType = item.getNewsType();
            // 处理h5文本
            String context = null;
            if (CustomerCommonConstant.REPLACE_LINK_NEWS_TYPE_LIST.contains(newsType)) {
                context = CustomerCommonConstant.CUSTOMER_TEXT_LINK_TEMPLATE.replace(CustomerCommonConstant.CUSTOMER_TEXT_CONTENT_PLACEHOLDER, item.getTextContent());
            } else if (CustomerCommonConstant.COMMON_NEWS_TYPE_LIST.contains(newsType)){
                context = item.getTextContent();
            }
            if (ObjectUtil.isNotNull(context)) {
                // 拼接的要替换链接
                mainH5Content.append(context.replace(CustomerCommonConstant.CUSTOMER_TEXT_URL_PLACEHOLDER, item.getReferral()));
                if (idx != referralEntityList.size()) {
                    mainH5Content.append("\n").append("\n");
                }
                // 将链接中的占位符换成链接
                customerContentVO.setContent(context.replace(CustomerCommonConstant.CUSTOMER_TEXT_URL_PLACEHOLDER + idx, "#"));
            }

            // 赋值链接信息
            CustomerReferralDTO customerReferralDto = new CustomerReferralDTO();
            BeanUtil.copyProperties(item, customerReferralDto);

            customerContentVO.setCustomerReferralDto(customerReferralDto);
            contentList.add(customerContentVO);

            idx += 1;
        }
        customerGraphicsDelay.setContent(mainH5Content.toString().replace(CustomerCommonConstant.H5_STYLE_CODE, ""));
        detailVO.setContentList(contentList);

        // 公众号
        AuthInfoVO authInfoVO = super.baseMapper.getCustomerDelayAuthInfo(id);
        detailVO.setAuthInfoVo(authInfoVO);
        // 更新连接表
        localLog.info("更新连接表");
        if (CollectionUtil.isNotEmpty(referralEntityList)) {
            referralEntityService.updateBatchById(referralEntityList);
        }
        // 更新主表
        localLog.info("更新主表");
        customerGraphicsDelay.initOperateInfo(tokenManager.getUserIdFromToken(), false);
        super.updateById(customerGraphicsDelay);

        return new GenericsResult<>(detailVO);
    }

    @Override
    public GenericsResult<CustomerDelayTextDetailVO> removeCustomerContent(Long contentId) {
        ReferralEntity referralEntity = referralEntityService.getById(contentId);
        if (ObjectUtil.isNull(referralEntity)) {
            return new GenericsResult<>(false, "找不到主体数据");
        }
        boolean result = referralEntityService.removeById(contentId);
        if (result) {
            return getCustomerTextDetail(referralEntity.getMaterialGraphicsId());
        }
        return new GenericsResult<>(false, "删除失败");
    }
}
