package com.yaoyaozw.customer.listener;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.yaoyaozw.customer.constants.CustomerMaterialConstant;
import com.yaoyaozw.customer.dto.AcquisitionExceptionHandleParam;
import com.yaoyaozw.customer.dto.WeChatTextDTO;
import com.yaoyaozw.customer.entity.AuthorizerInfo;
import com.yaoyaozw.customer.entity.CustomerFollowReply;
import com.yaoyaozw.customer.entity.CustomerFollowReplyMultiNews;
import com.yaoyaozw.customer.entity.ReferralEntity;
import com.yaoyaozw.customer.event.AcquisitionExceptionEvent;
import com.yaoyaozw.customer.service.CustomerFollowReplyMultiNewsService;
import com.yaoyaozw.customer.service.CustomerFollowReplyService;
import com.yaoyaozw.customer.service.CustomerFollowReplyService;
import com.yaoyaozw.customer.service.ReferralEntityService;
import com.yaoyaozw.customer.utils.WeChatUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.ApplicationListener;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@Component
public class FollowReplyAcquisitionListener implements ApplicationListener<AcquisitionExceptionEvent> {


    @Resource
    private CustomerFollowReplyService customerFollowReplyService;
    @Resource
    private CustomerFollowReplyMultiNewsService customerFollowReplyMultiNewsService;
    @Resource
    private ReferralEntityService referralEntityService;


    @Override
    public void onApplicationEvent(AcquisitionExceptionEvent acquisitionExceptionEvent) {
        AcquisitionExceptionHandleParam source = acquisitionExceptionEvent.getSource();
        if (source == null) {
            return;
        }
        AuthorizerInfo authorizerInfo = source.getAuthorizerInfo();
        String fromPath = source.getFromPath();
        String toPath = source.getToPath();
        try {
            // 查询这个公众号延时客服用到了 fromPath 的地方，替换成 toPath
            List<CustomerFollowReply> allCustomerFollowReplyList = customerFollowReplyService.list(new QueryWrapper<CustomerFollowReply>().eq("appid", authorizerInfo.getAppid()));

            // 过滤出发送链接是 fromPath 或者发送文本中的跳转链接包含 fromPath 的延时客服配置
            List<CustomerFollowReply> containsFromPathGraphicsList = allCustomerFollowReplyList.stream().filter(v -> (StringUtils.isNotBlank(v.getSourceUrl()) && v.getSourceUrl().contains(fromPath)) || (StringUtils.isNotBlank(v.getContent()) && v.getContent().contains(fromPath))).collect(Collectors.toList());
            // 多图文
            List<Long> multiNewsReplyIds = allCustomerFollowReplyList.stream().filter(v -> CustomerMaterialConstant.TENCENT_MEDIA_TYPE_MULTI_NEWS.equals(v.getType())).map(CustomerFollowReply::getId).collect(Collectors.toList());
            List<CustomerFollowReplyMultiNews> customerFollowReplyMultiNewsList = new ArrayList<>();
            if (CollectionUtil.isNotEmpty(multiNewsReplyIds)) {
                customerFollowReplyMultiNewsList = customerFollowReplyMultiNewsService.list(new QueryWrapper<CustomerFollowReplyMultiNews>().in("reply_id", multiNewsReplyIds));
            }

            // 查询链接配置
            List<Long> graphicsIds = containsFromPathGraphicsList.stream().map(CustomerFollowReply::getId).collect(Collectors.toList());
            customerFollowReplyMultiNewsList.forEach(v -> graphicsIds.add(v.getId()));
            List<ReferralEntity> relatedReferralEntityList = new ArrayList<>();
            if (CollectionUtil.isNotEmpty(graphicsIds)) {
                relatedReferralEntityList = referralEntityService.list(new QueryWrapper<ReferralEntity>().in(ReferralEntity.COL_MATERIAL_GRAPHICS_ID, graphicsIds));
            }
            // 替换 graphics 和 referral 中的 fromPath 为 toPath
            List<CustomerFollowReply> updateGraphicsList = allCustomerFollowReplyList.stream().filter(item -> {
                boolean update = false;
                // 直跳链接
                if (StringUtils.isNotBlank(item.getSourceUrl()) && item.getSourceUrl().contains(fromPath)) {
                    item.setSourceUrl(item.getSourceUrl().replaceAll(fromPath, toPath));
                    update = true;
                }
                // 文本链接
                if (StringUtils.isNotBlank(item.getContent()) && item.getContent().contains(fromPath)) {
                    item.setContent(item.getContent().replaceAll(fromPath, toPath));
                    update = true;
                }
                return update;
            }).collect(Collectors.toList());

            List<CustomerFollowReplyMultiNews> updateMultiNewsList = customerFollowReplyMultiNewsList.stream().filter(item -> {
                boolean update = false;
                if (StringUtils.isNotBlank(item.getLinkUrl()) && item.getLinkUrl().contains(fromPath)) {
                    item.setLinkUrl(item.getLinkUrl().replaceAll(fromPath, toPath));
                    update = true;
                }
                return update;
            }).collect(Collectors.toList());

            List<ReferralEntity> updateReferralList = relatedReferralEntityList.stream().filter(item -> {
                boolean update = false;
                // 直跳链接
                if (StringUtils.isNotBlank(item.getReferral()) && item.getReferral().contains(fromPath)) {
                    item.setReferral(item.getReferral().replaceAll(fromPath, toPath));
                    update = true;
                }
                // 文本链接
                if (StringUtils.isNotBlank(item.getTextContent()) && item.getTextContent().contains(fromPath)) {
                    item.setTextContent(item.getTextContent().replaceAll(fromPath, toPath));
                    update = true;
                }
                return update;
            }).collect(Collectors.toList());

            // 执行更新
            boolean update = false;
            if (CollectionUtil.isNotEmpty(updateGraphicsList)) {
                // 更新主配置
                customerFollowReplyService.updateBatchById(updateGraphicsList);
                update = true;
                customerFollowReplyService.putMaterialToRedis(authorizerInfo.getAppid(), null);
            }
            if (CollectionUtil.isNotEmpty(updateMultiNewsList)) {
                // 更新多图文配置
                customerFollowReplyMultiNewsService.updateBatchById(updateMultiNewsList);
                update = true;
                customerFollowReplyMultiNewsService.putMaterialToRedis(authorizerInfo.getAppid(), customerFollowReplyMultiNewsList);
            }
            if (CollectionUtil.isNotEmpty(updateReferralList)) {
                // 更新链接配置
                referralEntityService.updateBatchById(updateReferralList);
                update = true;
            }
            if (update) {
                // 发送报警信息
                String msg = "【关回-获客链接替换】公众号：" + authorizerInfo.getNickName() + " 获客链接：" + fromPath + " 已被替换为：" + toPath;
                WeChatTextDTO cover = new WeChatTextDTO().cover(msg, null, null);
                WeChatUtils.sendErrorMessage(cover, "9934244e-9694-4a0c-95dd-ddd6ca4bd959");
            } else {
                log.info("公众号：{} 暂无使用获客链接：{} 的关回配置", authorizerInfo.getAppid(), fromPath);
            }
        } catch (Exception e) {
            log.error("公众号：{} 关回链接从：{} 替换为：{} 异常：{}", source.getAuthorizerInfo().getNickName(), source.getFromPath(), source.getToPath(), e.getMessage(), e);
        }

    }
}
