package com.yaoyaozw.customer.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yaoyaozw.customer.common.BaseResult;
import com.yaoyaozw.customer.common.GenericsResult;
import com.yaoyaozw.customer.components.FollowReplyComponent;
import com.yaoyaozw.customer.components.SnowflakeComponent;
import com.yaoyaozw.customer.constants.CustomerMaterialConstant;
import com.yaoyaozw.customer.dto.follow.FollowReplyCopyDTO;
import com.yaoyaozw.customer.dto.follow.FollowReplyMultiNewsCreateDTO;
import com.yaoyaozw.customer.dto.follow.FollowReplyQueryDTO;
import com.yaoyaozw.customer.dto.follow.FollowReplySaveDTO;
import com.yaoyaozw.customer.entity.CommonReferralBody;
import com.yaoyaozw.customer.entity.CustomerFollowReply;
import com.yaoyaozw.customer.entity.CustomerFollowReplyMultiNews;
import com.yaoyaozw.customer.entity.ReferralEntity;
import com.yaoyaozw.customer.mapper.CustomerFollowReplyMapper;
import com.yaoyaozw.customer.service.AuthorizerInfoService;
import com.yaoyaozw.customer.service.CustomerFollowReplyMultiNewsService;
import com.yaoyaozw.customer.service.CustomerFollowReplyService;
import com.yaoyaozw.customer.service.ReferralEntityService;
import com.yaoyaozw.customer.utils.TencentCustomerUtil;
import com.yaoyaozw.customer.vo.AuthInfoVO;
import com.yaoyaozw.customer.vo.TencentMediaResponseVO;
import com.yaoyaozw.customer.vo.follow.FollowReplyCopyResultVO;
import com.yaoyaozw.customer.vo.follow.FollowReplyInfoVO;
import com.yaoyaozw.customer.vo.follow.FollowReplyListVO;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author darker
 * @date 2023/3/14 11:15
 */
@Service
public class CustomerFollowReplyServiceImpl extends ServiceImpl<CustomerFollowReplyMapper, CustomerFollowReply> implements CustomerFollowReplyService {

    private final static Logger localLog = LoggerFactory.getLogger(CustomerFollowReplyServiceImpl.class);

    @Autowired
    private TencentCustomerUtil tencentCustomerUtil;
    @Autowired
    private SnowflakeComponent snowflakeComponent;
    @Autowired
    private FollowReplyComponent followReplyComponent;
    @Autowired
    private ReferralEntityService referralEntityService;
    @Autowired
    private AuthorizerInfoService authorizerInfoService;
    @Autowired
    private RedisTemplate<String, Object> redisTemplate;
    @Autowired
    private CustomerFollowReplyMultiNewsService followReplyMultiNewsService;


    @Override
    public GenericsResult<List<FollowReplyListVO>> list(FollowReplyQueryDTO queryDto) {
        List<FollowReplyListVO> list = this.baseMapper.getList(queryDto);
        return new GenericsResult<>(list);
    }

    @Override
    public GenericsResult<String> create(FollowReplySaveDTO saveDto) {
        String result = checkNecessary(saveDto);
        if (StringUtils.isNotEmpty(result)) {
            return new GenericsResult<>(false, result);
        }

        CustomerFollowReply entity = new CustomerFollowReply();
        BeanUtil.copyProperties(saveDto, entity);

        CustomerFollowReply sourceEntity = null;
        boolean isCreate = ObjectUtil.isNull(entity.getId());
        if (!isCreate) {
            sourceEntity = this.getById(entity.getId());
            if (ObjectUtil.isNull(sourceEntity)) {
                return new GenericsResult<>(false, "");
            }
        }
        if (isCreate) {
            // 新增的id
            entity.setId(snowflakeComponent.snowflakeId());
        } else if (!entity.getType().equals(sourceEntity.getType()) && CustomerMaterialConstant.needReferral(sourceEntity.getType())) {
            // 如果编辑更换过类型，清除掉referral数据
            referralEntityService.remove(new QueryWrapper<ReferralEntity>().eq(ReferralEntity.COL_MATERIAL_GRAPHICS_ID, sourceEntity.getId()));
        }

        entity.setStatus(1);
        // 判断是否需要上传文件(图片、语音)
        if (CustomerMaterialConstant.needUpload(entity.getType())) {
            // 编辑的时候，删除原来的media
            if (!isCreate) {
                if (StringUtils.isNotBlank(sourceEntity.getTxMediaId())) {
                    TencentMediaResponseVO removeResponse = tencentCustomerUtil.removePermanentMedia(sourceEntity.getAppid(), sourceEntity.getTxMediaId());
                    if (ObjectUtil.isNull(removeResponse.getErrcode()) || removeResponse.getErrcode() != 0) {
                        return new GenericsResult<>(false, "删除原本素材失败: " + removeResponse.getErrmsg());
                    }
                }
            }
            // 上传文件
            TencentMediaResponseVO uploadResponse = tencentCustomerUtil.uploadTencentMedia(entity.getAppid(), entity.getOriginMediaUrl(), entity.getType());
            if (StringUtils.isNotBlank(uploadResponse.getErrmsg())) {
                // 上传失败
                return new GenericsResult<>(false, uploadResponse.getErrmsg());
            }
            entity.setTxMediaUrl(uploadResponse.getUrl());
            entity.setTxMediaId(uploadResponse.getMedia_id());
        } else if (CustomerMaterialConstant.TENCENT_MEDIA_TYPE_MULTI_NEWS.equals(entity.getType())) {
            // 多图文
            List<FollowReplyMultiNewsCreateDTO> multiNewsList = saveDto.getMultiNewsList();
            localLog.info("appId {} follow reply has multi news size {}", saveDto.getAppid(), multiNewsList.size());
            followReplyMultiNewsService.saveFollowReplyMultiNews(entity.getId(), saveDto.getAccountEntity(), saveDto.getMultiNewsList());
//            return new GenericsResult<>(true, "【测试】多图文关回设置成功");
        } else if (!CustomerMaterialConstant.TENCENT_MEDIA_TYPE_TEXT.equals(entity.getType())){
            // 获取书城公众号链接(图文)
            ReferralEntity referralEntity;
            try {
                referralEntity = followReplyComponent.getCreateReferralEntity(saveDto.getReferralBody());
            } catch (Exception e) {
                throw new RuntimeException("获取链接异常: " + e.getMessage());
            }
            referralEntity.setMaterialGraphicsId(entity.getId());
            entity.setSourceUrl(referralEntity.getReferral());
            entity.setContent("");
            // 保存链接数据
            if (ObjectUtil.isNull(referralEntity.getId())) {
                referralEntityService.save(referralEntity);
            } else {
                referralEntityService.updateById(referralEntity);
            }
        }
        // 保存主体数据
        if (isCreate) {
            this.save(entity);
        } else {
            this.updateById(entity);
        }
        putMaterialToRedis(entity.getAppid(), null);
        return new GenericsResult<>(String.valueOf(entity.getId()));
    }

    @Override
    public GenericsResult<List<CommonReferralBody>> createTextItem(CommonReferralBody referralBody) {
        CustomerFollowReply entity = this.getById(referralBody.getMaterialGraphicsId());
        if (ObjectUtil.isNull(entity)) {
            return new GenericsResult<>(false, "找不到主体数据");
        }
        ReferralEntity referralEntity;
        try {
            referralEntity = followReplyComponent.getCreateReferralEntity(referralBody);
        } catch (Exception e) {
            throw new RuntimeException("获取链接异常: " + e.getMessage());
        }
        // 保存链接数据
        if (ObjectUtil.isNull(referralEntity.getId())) {
            referralEntityService.save(referralEntity);
        } else {
            referralEntityService.updateById(referralEntity);
        }
        List<ReferralEntity> referralList = referralEntityService.list(new QueryWrapper<ReferralEntity>().eq(ReferralEntity.COL_MATERIAL_GRAPHICS_ID, entity.getId()).orderByAsc(ReferralEntity.COL_GMT_CREATE));

        String content = followReplyComponent.reSortAndSaveTextItem(referralList);

        entity.setContent(content);

        // 保存主体
        this.updateById(entity);
        return new GenericsResult<>(getReferralBodyFromEntity(referralList));
    }

    @Override
    public BaseResult remove(Long id) {
        // 从腾讯删除文件
        CustomerFollowReply entity = this.getById(id);
        if (ObjectUtil.isNull(entity)) {
            return new BaseResult().error("找不到实体数据");
        }
        if (StringUtils.isNotEmpty(entity.getTxMediaId())) {
            tencentCustomerUtil.removePermanentMedia(entity.getAppid(), entity.getTxMediaId());
        }

        // 删除本体
        this.removeById(id);
        // 删除相关链接
        referralEntityService.remove(new QueryWrapper<ReferralEntity>().eq(ReferralEntity.COL_MATERIAL_GRAPHICS_ID, id));
        // 重置redis中的素材
        putMaterialToRedis(entity.getAppid(), null);

        return new BaseResult().success("删除");
    }

    @Override
    public GenericsResult<List<CommonReferralBody>> removeTextItem(Long id) {
        ReferralEntity referralEntity = referralEntityService.getById(id);
        if (ObjectUtil.isNull(referralEntity)) {
            return new GenericsResult<>(false, "未找到该文本子素材");
        }
        // 查询子素材所属素材
        CustomerFollowReply entity = this.getById(referralEntity.getMaterialGraphicsId());
        if (ObjectUtil.isNull(entity)) {
            return new GenericsResult<>(false, "未找到该文本所属素材");
        }
        // 删除子文本
        referralEntityService.removeById(id);
        // 重新排序构造
        List<ReferralEntity> referralList = referralEntityService.list(new QueryWrapper<ReferralEntity>().eq(ReferralEntity.COL_MATERIAL_GRAPHICS_ID, referralEntity.getMaterialGraphicsId()).orderByAsc(ReferralEntity.COL_GMT_CREATE));
        String content = followReplyComponent.reSortAndSaveTextItem(referralList);
        entity.setContent(content);
        // 保存实体
        this.updateById(entity);
        return new GenericsResult<>(getReferralBodyFromEntity(referralList));
    }

    @Override
    public BaseResult copy(FollowReplyCopyDTO copyDto) {
        AuthInfoVO sourceAuth = copyDto.getSourceAuth();
        List<AuthInfoVO> targetAuthList = copyDto.getTargetAuthList();

        long copyFlag = System.currentTimeMillis();
        localLog.info("关回复用, 复用批次号: {}, 源公众号: {}, 目标公众号: {}条", copyFlag, sourceAuth.getAccountName(), targetAuthList.size());
        List<CustomerFollowReply> sourceMaterialList = getSourceMaterialList(sourceAuth.getAppId());
        if (CollectionUtil.isEmpty(sourceMaterialList)) {
            return new BaseResult().error("源公众号找不到素材");
        }
        // 删除目标公众号现有的素材和链接
        removeOriginMaterialList(targetAuthList);

        List<String> errorAuthList = null;
        int idx = 1;
        for (AuthInfoVO targetAuth : targetAuthList) {
            localLog.info("批次号: {}, 公众号处理进度: {}/{}", copyFlag, idx++, targetAuthList.size());
            // 调用复用
            FollowReplyCopyResultVO result = followReplyComponent.copyMaterialToTarget(sourceAuth.getAccountName(), targetAuth, sourceMaterialList);
            if (result.getHasError()) {
                if (errorAuthList == null) {
                    errorAuthList = new ArrayList<>();
                }
                errorAuthList.add(targetAuth.getAccountName());
            } else {
                if (CollectionUtil.isNotEmpty(result.getMaterialList())) {
                    this.saveBatch(result.getMaterialList());
                    // 复用之后的结果存储到 redis
                    putMaterialToRedis(targetAuth.getAppId(), result.getMaterialList());
                }
                if (CollectionUtil.isNotEmpty(result.getMultiNewsList())) {
                    followReplyMultiNewsService.saveBatch(result.getMultiNewsList());
                }
                if (CollectionUtil.isNotEmpty(result.getReferralEntityList())) {
                    referralEntityService.saveBatch(result.getReferralEntityList());
                }

            }
        }
        if (CollectionUtil.isNotEmpty(errorAuthList)) {
            return new BaseResult().error("部分成功; 复用异常公众号: " + errorAuthList);
        }

        return new BaseResult().success("复用完成");
    }

    @Override
    public GenericsResult<FollowReplyInfoVO> getInfo(Long id) {

        CustomerFollowReply entity = this.getById(id);
        FollowReplyInfoVO infoVo = new FollowReplyInfoVO();

        BeanUtil.copyProperties(entity, infoVo);
        if (!CustomerMaterialConstant.TENCENT_MEDIA_TYPE_MULTI_NEWS.equals(entity.getType())) {
            // 获取素材的链接数据
            if (CustomerMaterialConstant.needReferral(entity.getType())) {
                // 是需要设置referral的类型
                if (CustomerMaterialConstant.TENCENT_MEDIA_TYPE_TEXT.equals(entity.getType())) {
                    // 文本类型
                    List<ReferralEntity> referralList = referralEntityService.list(new QueryWrapper<ReferralEntity>().eq(ReferralEntity.COL_MATERIAL_GRAPHICS_ID, id).orderByAsc("sort"));
                    infoVo.setTextBodyList(getReferralBodyFromEntity(referralList));
                } else if (CustomerMaterialConstant.TENCENT_MEDIA_TYPE_NEWS.equals(entity.getType())) {
                    try {
                        // 设置图文类型的链接数据
                        ReferralEntity referralEntity = referralEntityService.getOne(new QueryWrapper<ReferralEntity>().eq(ReferralEntity.COL_MATERIAL_GRAPHICS_ID, id));
                        if (ObjectUtil.isNotNull(referralEntity)) {
                            CommonReferralBody referralBody = new CommonReferralBody();
                            BeanUtil.copyProperties(referralEntity, referralBody);
                            infoVo.setReferralBody(referralBody);
                        }
                    } catch (Exception e) {
                        return new GenericsResult<>(false, "获取图文链接实体异常");
                    }
                }
            }
        } else {
            // 获取多图文
            List<CustomerFollowReplyMultiNews> multiNewsList = followReplyMultiNewsService.list(new QueryWrapper<CustomerFollowReplyMultiNews>().eq("reply_id", id));
            if (CollectionUtil.isNotEmpty(multiNewsList)) {
                List<Long> multiNewsIdList = multiNewsList.stream().map(CustomerFollowReplyMultiNews::getId).collect(Collectors.toList());
                List<ReferralEntity> referralEntityList = referralEntityService.list(new QueryWrapper<ReferralEntity>().in(ReferralEntity.COL_MATERIAL_GRAPHICS_ID, multiNewsIdList));
                Map<Long, ReferralEntity> referralEntityMap = referralEntityList.stream().collect(Collectors.toMap(ReferralEntity::getMaterialGraphicsId, Function.identity(), (o1, o2) -> o2));
                multiNewsList.forEach(item -> item.setReferralEntity(referralEntityMap.get(item.getId())));
                infoVo.setMultiNewsList(multiNewsList);
            }
        }
        // 根据appid获取公众号信息
        AuthInfoVO authInfo = authorizerInfoService.getAuthInfoByAppid(infoVo.getAppid());
        if (ObjectUtil.isNull(authInfo)) {
            return new GenericsResult<>(false, "无法获取公众号信息");
        }
        infoVo.setAuthInfo(authInfo);
        return new GenericsResult<>(infoVo);
    }

    private List<CommonReferralBody> getReferralBodyFromEntity(List<ReferralEntity> referralList) {
        followReplyComponent.contractH5ContentBatch(referralList);
        JSONArray referralJsonArray = JSONUtil.parseArray(referralList);
        return JSONUtil.toList(referralJsonArray, CommonReferralBody.class);
    }

    private List<CustomerFollowReply> getSourceMaterialList(String appid) {
        List<CustomerFollowReply> sourceMaterialList = this.list(new QueryWrapper<CustomerFollowReply>().eq("appid", appid));

        if (CollectionUtil.isEmpty(sourceMaterialList)) {
            return null;
        }
        localLog.info("获取到源公众号素材: {}条", sourceMaterialList.size());
        // 获取链接
        List<Long> materialIdList = sourceMaterialList.stream().map(CustomerFollowReply::getId).collect(Collectors.toList());
        List<ReferralEntity> referralEntityList = referralEntityService.list(new QueryWrapper<ReferralEntity>().in(ReferralEntity.COL_MATERIAL_GRAPHICS_ID, materialIdList));
        // 查询是否存在多图文关回配置
        List<CustomerFollowReplyMultiNews> multiNewsList = followReplyMultiNewsService.list(new QueryWrapper<CustomerFollowReplyMultiNews>().in("reply_id", materialIdList));
        Map<Long, List<CustomerFollowReplyMultiNews>> multiNewsListMap = new HashMap<>(4);
        if (CollectionUtil.isNotEmpty(multiNewsList)) {
            // 查询这些多图文配置及其链接列表
            List<Long> multiNewsIdList = multiNewsList.stream().map(CustomerFollowReplyMultiNews::getId).collect(Collectors.toList());
            List<ReferralEntity> multiNewsReferralEntityList = referralEntityService.list(new QueryWrapper<ReferralEntity>().in(ReferralEntity.COL_MATERIAL_GRAPHICS_ID, multiNewsIdList));
            Map<Long, ReferralEntity> referralEntityMap = multiNewsReferralEntityList.stream().collect(Collectors.toMap(ReferralEntity::getMaterialGraphicsId, Function.identity(), (o1, o2) -> o2));
            // 匹配多图文与链接体
            multiNewsListMap = multiNewsList.stream().peek(item -> item.setReferralEntity(referralEntityMap.get(item.getId()))).collect(Collectors.groupingBy(CustomerFollowReplyMultiNews::getReplyId));
        }

        localLog.info("获取到源公众号链接数据: {}条", referralEntityList.size());
        Map<Long, List<ReferralEntity>> referralMap = referralEntityList.stream().collect(Collectors.groupingBy(ReferralEntity::getMaterialGraphicsId));

        // 分配referral
        for (CustomerFollowReply sourceEntity : sourceMaterialList) {
            sourceEntity.setMultiNewsList(multiNewsListMap.get(sourceEntity.getId()));
            List<ReferralEntity> referralEntities = referralMap.get(sourceEntity.getId());
            if (CollectionUtil.isNotEmpty(referralEntities)) {
                referralEntities = referralEntities.stream().sorted(Comparator.comparingInt(ReferralEntity::getSort)).collect(Collectors.toList());
                sourceEntity.setReferralEntityList(referralEntities);
            }
        }

        return sourceMaterialList;
    }

    private void removeOriginMaterialList(List<AuthInfoVO> targetList) {
        List<String> targetAppidList = targetList.stream().map(AuthInfoVO::getAppId).collect(Collectors.toList());

        List<CustomerFollowReply> sourceMaterialList = this.list(new QueryWrapper<CustomerFollowReply>().in("appid", targetAppidList));
        // 原来没有
        if (CollectionUtil.isEmpty(sourceMaterialList)) {
            return;
        }
        List<Long> sourceMaterialIdList = new ArrayList<>();

        for (CustomerFollowReply sourceMaterial : sourceMaterialList) {
            sourceMaterialIdList.add(sourceMaterial.getId());
            if (StringUtils.isNotEmpty(sourceMaterial.getTxMediaId())) {
                // 删除公众号后台的素材
                tencentCustomerUtil.removePermanentMedia(sourceMaterial.getAppid(), sourceMaterial.getTxMediaId());
            }
        }

        // 删除主体
        this.removeByIds(sourceMaterialIdList);
        // 删除链接
        referralEntityService.remove(new QueryWrapper<ReferralEntity>().in(ReferralEntity.COL_MATERIAL_GRAPHICS_ID, sourceMaterialIdList));

    }

    private String checkNecessary(FollowReplySaveDTO saveDto) {
        if (!CustomerMaterialConstant.TENCENT_MEDIA_TYPE_TEXT.equals(saveDto.getType()) && !CustomerMaterialConstant.TENCENT_MEDIA_TYPE_MULTI_NEWS.equals(saveDto.getType())) {
            // 文本类型和多图文类型的都不需要文件
            if (StringUtils.isEmpty(saveDto.getOriginMediaUrl())) {
                return "请选择素材文件";
            }
        }


        return null;
    }

    private void putMaterialToRedis(String appid, List<CustomerFollowReply> entityList) {
        if (StringUtils.isNotEmpty(appid) && CollectionUtil.isEmpty(entityList)) {
            // 传参没传实体, 现查
            entityList = this.list(new QueryWrapper<CustomerFollowReply>().eq("appid", appid));
        }
        if (CollectionUtil.isNotEmpty(entityList)) {
            String jsonStr = JSONUtil.toJsonStr(entityList);
            redisTemplate.opsForHash().put(CustomerMaterialConstant.FOLLOW_REPLY_REDIS_KEY, appid, jsonStr);
        } else {
            redisTemplate.opsForHash().delete(CustomerMaterialConstant.FOLLOW_REPLY_REDIS_KEY, appid);
        }
    }

}
