package com.yaoyaozw.customer.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.yaoyaozw.customer.dto.AcquisitionExceptionHandleParam;
import com.yaoyaozw.customer.entity.AuthorizerInfo;
import com.yaoyaozw.customer.entity.CompanyAcquisitionLink;
import com.yaoyaozw.customer.mapper.CompanyAcquisitionLinkMapper;
import com.yaoyaozw.customer.mapper.MaterialCommonMapper;
import com.yaoyaozw.customer.service.AuthorizerInfoService;
import com.yaoyaozw.customer.service.CompanyAcquisitionLinkService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Slf4j
@Service
public class CompanyAcquisitionLinkServiceImpl implements CompanyAcquisitionLinkService {

    public static final String ACQUISITION_REPLACE_RECORD = "operate:customer:acquisition:exception:replace:record";

    @Resource
    private MaterialCommonMapper materialCommonMapper;
    @Resource
    private CompanyAcquisitionLinkMapper companyAcquisitionLinkMapper;
    @Resource
    private AuthorizerInfoService authorizerInfoService;
    @Resource
    private RedisTemplate<String, Object> redisTemplate;

    @Override
    public String checkAndSearchTargetReferral(String sourceReferral, Long targetAuthId) {
        // 查询 sourceReferral 这个链接是不是获客链接，如果不是，直接返回原链接
        String type = materialCommonMapper.getAppointReferralType(sourceReferral);
        if (StringUtils.isBlank(type)) {
            return sourceReferral;
        }
        // 原链接是获客链接，查询目标公众号的获客链接
        List<String> acquisitionLinkList = materialCommonMapper.getAcquisitionLinkList(targetAuthId, type);
        if (CollectionUtil.isEmpty(acquisitionLinkList)) {
            throw new RuntimeException("目标公众号无法获取获客链接");
        }
/*        if (acquisitionLinkList.size() > 1) {
            throw new RuntimeException("目标公众号存在多个有效获客链接");
        }*/
        return acquisitionLinkList.get(0);
    }

    @Override
    public AcquisitionExceptionHandleParam getReplaceAcquisitionEntityByLinkId(String linkId) {
        // 查询链接绑定公众号信息
        CompanyAcquisitionLink acquisitionInfo;
        try {
            acquisitionInfo = companyAcquisitionLinkMapper.getAcquisitionInfoByLinkId(linkId);
        } catch (Exception e) {
            log.error("获取链接Id：{} 信息异常", linkId, e);
            return null;
        }
        if (acquisitionInfo == null) {
            log.error("无法获取链接Id：{} 信息", linkId);
            return null;
        }
        AuthorizerInfo authorizerInfo;
        try {
            authorizerInfo = authorizerInfoService.getOne(new QueryWrapper<AuthorizerInfo>().eq("appid", acquisitionInfo.getAuthAppId()));
        } catch (Exception e) {
            log.error("获取公众号信息异常：{}", acquisitionInfo.getAuthAppId(), e);
            return null;
        }
        if (authorizerInfo == null) {
            log.error("无法获取公众号信息：{}", acquisitionInfo.getAuthAppId());
            return null;
        }
        // 查询可替换使用的链接列表
        List<CompanyAcquisitionLink> availableAcquisitionList = companyAcquisitionLinkMapper.getAvailableAcquisitionListExceptLinkId(acquisitionInfo.getAuthAppId(), linkId, acquisitionInfo.getType());
        if (CollectionUtil.isEmpty(availableAcquisitionList)) {
            log.error("公众号：{} 无法获取可替换的链接", acquisitionInfo.getAuthAppId());
            return null;
        }
        try {
            // 查询替换记录
            Object recordVal = redisTemplate.opsForHash().get(ACQUISITION_REPLACE_RECORD, acquisitionInfo.getAuthAppId());
            List<String> allExecutedLinkIds = new ArrayList<>();
            if (recordVal != null) {
                List<String> executedLinkIds = JSONUtil.toList(JSONUtil.parseArray(recordVal), String.class);
                // 收集已执行记录
                allExecutedLinkIds.addAll(executedLinkIds);
                // 顾虑掉已经执行过替换的链接
                List<CompanyAcquisitionLink> afterFilterAvailableAcquisitionList = availableAcquisitionList.stream().filter(v -> !executedLinkIds.contains(v.getLinkId())).collect(Collectors.toList());
                if (CollectionUtil.isNotEmpty(afterFilterAvailableAcquisitionList)) {
                    // 如果筛选之后还有值，则使用筛选之后的列表
                    availableAcquisitionList = afterFilterAvailableAcquisitionList;
                } else {
                    // 如果筛选之后没有值了，清除缓存从头再来
                    redisTemplate.opsForHash().delete(ACQUISITION_REPLACE_RECORD, acquisitionInfo.getAuthAppId());
                    allExecutedLinkIds = new ArrayList<>();
                    TimeUnit.SECONDS.sleep(1);
                }
            }
            // 记录当前的异常链接，存入缓存
            allExecutedLinkIds.add(acquisitionInfo.getLinkId());
            redisTemplate.opsForHash().put(ACQUISITION_REPLACE_RECORD, acquisitionInfo.getAuthAppId(), JSONUtil.toJsonStr(allExecutedLinkIds));
        } catch (Exception e) {
            log.error("公众号：{} 原链接Id：{} 过滤已执行链接异常：", acquisitionInfo.getAuthAppId(), acquisitionInfo.getLinkId(), e);
        }

        return new AcquisitionExceptionHandleParam(authorizerInfo, acquisitionInfo.getLinkUrl(), availableAcquisitionList.get(0).getLinkUrl());
    }

    @Override
    public List<String> getAcquisitionLinksByAppId(String appId, String type) {
        try {
            return materialCommonMapper.getAcquisitionLinkListByAppIdAndType(appId, type);
        } catch (Exception e) {
            log.error("获取公众号：{} 类型：{} 的获客链接异常", appId, type, e);
            return new ArrayList<>();
        }
    }
}
