package com.yaoyaozw.customer.handler;

import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSONObject;
import com.yaoyaozw.customer.entity.AuthorizerExpandInfo;
import com.yaoyaozw.customer.entity.AuthorizerInfo;
import com.yaoyaozw.customer.entity.ReferralEntity;
import com.yaoyaozw.customer.feigns.ReferralFeignClient;
import com.yaoyaozw.customer.service.AuthorizerExpandInfoService;
import com.yaoyaozw.customer.service.AuthorizerInfoService;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Resource;

/**
 * 书城推广链接处理器抽象类
 *
 * @author system
 */
public abstract class MpReferralHandler {

    @Resource
    protected ReferralFeignClient referralFeignClient;

    @Resource
    protected AuthorizerInfoService authorizerInfoService;

    @Resource
    protected AuthorizerExpandInfoService authorizerExpandInfoService;

    /**
     * 查询推广链接（抽象方法，由子类实现）
     *
     * @param referralEntity 推广链接实体
     * @param authorizerInfo 授权方信息
     * @return 推广链接实体
     */
    public abstract ReferralEntity queryMpReferral(ReferralEntity referralEntity, AuthorizerInfo authorizerInfo);

    /**
     * 针对已经获取完成的 mpPath 字段，进行处理，处理之后的小程序链接必然会放置于mpPath字段；
     * 如果还需要其它处理，如：小程序链接的http链接，则会放于 referral 字段
     * @param referralEntity 原本处理并赋值好链接之后的实体
     * @param authorizerInfo 公众号信息
     * @param authorizerExpandInfo 公众号绑定信息
     * @param responseJson
     */
    public abstract void constructMpReferral(ReferralEntity referralEntity, AuthorizerInfo authorizerInfo, AuthorizerExpandInfo authorizerExpandInfo, JSONObject responseJson);

    /**
     * 查询推广链接（根据infoId查询相关信息后调用抽象方法）
     * 传入ReferralEntity不会修改其内部值，会复制新的ReferralEntity并返回
     * @param referralEntity 推广链接实体（包含infoId字段）
     * @return 推广链接实体
     */
    public ReferralEntity queryMpReferral(ReferralEntity referralEntity) {

        ReferralEntity neoReferralEntity = new ReferralEntity();
        BeanUtil.copyProperties(referralEntity, neoReferralEntity);

        // 根据infoId查询authorizer_info表
        Long infoId = neoReferralEntity.getInfoId();
        if (infoId == null) {
            throw new IllegalArgumentException("infoId不能为空");
        }
        AuthorizerInfo authorizerInfo = authorizerInfoService.getById(infoId);
        if (authorizerInfo == null) {
            throw new IllegalArgumentException("未找到对应的授权方信息，infoId: " + infoId);
        }

        // 根据appid查询authorizer_expand_info表（1条）
        String appid = authorizerInfo.getAppid();
        if (appid == null) {
            throw new IllegalArgumentException("公众号appid为空，infoId: " + infoId);
        }
        // 调用抽象方法
        return queryMpReferral(neoReferralEntity, authorizerInfo);
    }

    protected Boolean isMiniProgramPath(String path) {
        return StringUtils.isNotBlank(path) && (path.startsWith("page") || path.startsWith("/page"));
    }

    protected Boolean isHttpPath(String path) {
        return StringUtils.isNotBlank(path) && (path.startsWith("http"));
    }
}

