package com.yaoyaozw.customer.utils;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.yaoyaozw.customer.entity.AuthorizerToken;
import com.yaoyaozw.customer.entity.ReferralEntity;
import com.yaoyaozw.customer.service.AuthorizerTokenService;
import com.yaoyaozw.customer.service.ReferralEntityService;
import com.yaoyaozw.customer.vo.TencentMediaResponseVO;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author darker
 * @date 2023/3/13 20:11
 */
@Component
public class TencentCustomerUtil{

    private final static Logger localLog = LoggerFactory.getLogger(TencentCustomerUtil.class);

    private final static String AUTH_ACCESS_TOKEN_REDIS_KEY = "AUTH_ACCESS_TOKEN";
    private final static String MEDIA_ADD_URL = "https://api.weixin.qq.com/cgi-bin/material/add_material";
    private final static String MEDIA_DEL_URL = "https://api.weixin.qq.com/cgi-bin/material/del_material";

    @Autowired
    private AuthorizerTokenService authorizerTokenService;
    @Autowired
    private RedisTemplate<String, Object> redisTemplate;

    /**
     * 上传腾讯媒体
     *
     * @param appid 公众号
     * @param fileUrl     文件url
     * @return {@link TencentMediaResponseVO}
     */
    public TencentMediaResponseVO uploadTencentMedia(String appid, String fileUrl, String mediaType) {
        localLog.info("上传媒体文件, 原始文件地址: {}", fileUrl);
        // 获取公众号token
        String accessToken = getAccessTokenByAppid(appid);
        if (StringUtils.isEmpty(accessToken)) {
            localLog.warn("appid: {} 没找到accessToken", appid);
            TencentMediaResponseVO response = new TencentMediaResponseVO();
            response.setErrmsg("appid: " + appid + "没找到accessToken");
            return response;
        }
        // 根据文件地址获取文件
        String suffix = fileUrl.substring(fileUrl.lastIndexOf("=") + 1);
        byte[] fileByte = HttpClientUtil.readInputStream(fileUrl);

        // 构造参数，上传到腾讯后台
        Map<String, String> map = new HashMap<>(4);
        map.put("access_token", accessToken);
        map.put("type", mediaType);

        String json = HttpClientUtil.uploadFileByte(MEDIA_ADD_URL, fileByte, map, suffix);

        return JSONUtil.toBean(JSONUtil.parseObj(json), TencentMediaResponseVO.class);

    }

    /**
     * 删除永久媒体
     *
     * @param appid   appid
     * @param mediaId 媒体id
     * @return {@link TencentMediaResponseVO}
     */
    public TencentMediaResponseVO removePermanentMedia(String appid, String mediaId) {
        // 获取公众号token
        String accessToken = getAccessTokenByAppid(appid);
        // 构造参数，上传到腾讯后台
        Map<String, String> map = new HashMap<>(4);
        map.put("access_token", accessToken);
        JSONObject json = new JSONObject();
        json.put("media_id", mediaId);

        String jsonStr = JSONUtil.toJsonStr(json);

        String delResult = HttpClientUtil.doPostJson(MEDIA_DEL_URL, jsonStr, map);

        localLog.info("删除素材结果: {}", delResult);

        return JSONUtil.toBean(JSONUtil.parseObj(delResult), TencentMediaResponseVO.class);

    }

    /**
     * 通过appid获取访问令牌
     *
     * @param appid appid
     * @return {@link String}
     */
    private String getAccessTokenByAppid(String appid) {

        Object token = redisTemplate.opsForHash().get(AUTH_ACCESS_TOKEN_REDIS_KEY, appid);
        // 如果redis中没找到, 去数据库找
        if (ObjectUtil.isNull(token)) {
            AuthorizerToken authorizerToken = authorizerTokenService.findTokenByAppid(appid);
            return ObjectUtil.isNull(authorizerToken) ? null : authorizerToken.getAuthorizerAccessToken();
        }
        return String.valueOf(token);
    }

}
