package com.yaoyaozw.customer.service.impl;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.yaoyaozw.customer.common.BaseResult;
import com.yaoyaozw.customer.common.GenericsResult;
import com.yaoyaozw.customer.components.TokenManager;
import com.yaoyaozw.customer.dto.ConditionDTO;
import com.yaoyaozw.customer.dto.DistributeDTO;
import com.yaoyaozw.customer.dto.MenuMainCopyDTO;
import com.yaoyaozw.customer.dto.follow.FollowReplyCopyDTO;
import com.yaoyaozw.customer.dto.keyword.CustomerKeywordCopyDTO;
import com.yaoyaozw.customer.entity.AccountDistributeRecord;
import com.yaoyaozw.customer.feigns.MenuFeignClient;
import com.yaoyaozw.customer.mapper.AccountDistributeLogMapper;
import com.yaoyaozw.customer.mapper.AuthorizerTokenMapper;
import com.yaoyaozw.customer.mapper.KanbanCommonMapper;
import com.yaoyaozw.customer.service.*;
import com.yaoyaozw.customer.vo.AccountDistributeLogVO;
import com.yaoyaozw.customer.vo.AuthInfoVO;
import com.yaoyaozw.customer.vo.kanban.AccountSetupVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @Author: Dwl
 * @Date: 2024/03/27/17:22
 * @ClassName: DistributeServiceImpl
 * @Version: V3.2.0
 * @Description:
 */
@Slf4j
@Service
public class DistributeServiceImpl extends ServiceImpl<AccountDistributeLogMapper, AccountDistributeRecord> implements DistributeService {
    @Autowired
    private CustomerGraphicsDelayService customerGraphicsDelayService;

    @Autowired
    private CustomerKeywordService keywordService;

    @Autowired
    private CustomerFollowReplyService followReplyService;

    @Autowired
    private MenuFeignClient menuFeignClient;

    @Autowired
    private TokenManager tokenManager;
    @Autowired
    private AuthorizerTokenMapper authorizerTokenMapper;
    @Autowired
    private KanbanCommonMapper kanbanCommonMapper;
    @Autowired
    private MaterialMenuConfigService materialMenuConfigService;


    @Override
    public BaseResult distribute(DistributeDTO distributeDTO) {
        Long userId = tokenManager.getUserIdFromToken();
        List<AccountDistributeRecord> distributeLogList = new ArrayList<>();
        log.info("开始执行复用任务....");
        AuthInfoVO sourceAuth = distributeDTO.getSourceAuth();
        List<Long> channelIds = new ArrayList<>();
        List<String> nickNameList = new ArrayList<>();
        if (ObjectUtil.isNotNull(sourceAuth.getAppId())) {
            List<AccountSetupVO> accountSetupVOList = kanbanCommonMapper.getRole(userId, distributeDTO.getAccountId());
            if (accountSetupVOList.isEmpty()) {
                return new BaseResult().error("暂无权限");
            }
            channelIds = accountSetupVOList.stream().map(AccountSetupVO::getChannelId).collect(Collectors.toList());
            nickNameList = accountSetupVOList.stream().map(AccountSetupVO::getNickName).collect(Collectors.toList());
        }
        myExecutorDistribute(channelIds, distributeDTO, userId, distributeLogList, sourceAuth);

        return new BaseResult().success(JSONUtil.toJsonStr(nickNameList) + "：复用进行中, 请等待");
    }

    @Async("myExecutor")
    public void myExecutorDistribute(List<Long> channelIds, DistributeDTO distributeDTO, Long userId, List<AccountDistributeRecord> distributeLogList, AuthInfoVO sourceAuth) {
        List<String> targetAppList = distributeDTO.getTargetAppList();
        //获取到当天再投的进行复用
        List<AuthInfoVO> targetAuthList = distributeDTO.filterInfo(channelIds);
        try {
            if (!targetAppList.isEmpty()) {
                for (AuthInfoVO authInfoVO : targetAuthList) {
                    menuFeignClient.removeMenu(authInfoVO.getId(), authInfoVO.getAppId());
                }
            }
        } catch (Exception e) {
            log.error("Failed to delete menu configuration: {}", e.getMessage(), e);
        }

        //删除目标公众号配置
        try {
            List<Long> authIds = distributeDTO.getAuthIds();
            materialMenuConfigService.removeById(authIds);
        } catch (Exception e) {
            log.error("Failed to delete the target public account configuration :{}", e.getMessage(), e);
        }


        List<String> targetApp = targetAuthList.stream().map(AuthInfoVO::getAppId).collect(Collectors.toList());
        List<Long> idList = targetAuthList.stream().map(AuthInfoVO::getId).collect(Collectors.toList());
        List<String> typeList = distributeDTO.getTypeList();

        try {
            for (String type : typeList) {
                switch (type) {
                    case "1":
                        //关回复用
                        try {
                            FollowReplyCopyDTO followReplyCopyDTO = new FollowReplyCopyDTO(sourceAuth, targetAuthList);
                            followReplyService.copy(followReplyCopyDTO);
                        } catch (Exception e) {
                            log.error("Off reply failed: {}", e.getMessage(), e);
                        }

                        break;
                    case "2":
                        //关键词复用
                        try {
                            CustomerKeywordCopyDTO customerKeywordCopyDTO = new CustomerKeywordCopyDTO();
                            customerKeywordCopyDTO.replace(sourceAuth, targetAuthList, targetAppList);
                            keywordService.copy(customerKeywordCopyDTO);
                        } catch (Exception e) {
                            log.error("Keyword reuse failure: {}", e.getMessage(), e);
                        }
                        break;
                    case "3":
                        //延时客服复用
                        try {
                            customerGraphicsDelayService.copy(sourceAuth.getAppId(), "", "", null, targetApp);
                        } catch (Exception e) {
                            log.error("Delayed customer service reuse failure: {}", e.getMessage(), e);
                        }
                        break;
                    case "4":
                        //菜单管理复用
                        try {
                            MenuMainCopyDTO menuMainCopyDTO = new MenuMainCopyDTO();
                            menuMainCopyDTO.addList(sourceAuth.getId(), idList);
                            menuFeignClient.copy(menuMainCopyDTO);
                        } catch (Exception e) {
                            log.error("Menu management reuse failure: {}", e.getMessage(), e);
                        }
                        break;
                    default:
                        break;
                }
            }
        } catch (Exception e) {
            log.error("Distribution reuse exception：{}", e.getMessage(), e);
        }

        idList.forEach(c -> {
            AccountDistributeRecord distributeLog = new AccountDistributeRecord();
            distributeLog.saveDistribute(sourceAuth.getId(), c, typeList, userId);
            distributeLogList.add(distributeLog);
        });

        super.saveBatch(distributeLogList);
    }

    @Override
    public GenericsResult<PageInfo<AccountDistributeLogVO>> logList(ConditionDTO conditionDTO) {
        PageHelper.startPage(conditionDTO.getCurrentPage(), conditionDTO.getPageSize());
        List<AccountDistributeLogVO> accountDistributeLogVOList = this.baseMapper.selectLogList(conditionDTO);
        PageInfo<AccountDistributeLogVO> pageInfoList = new PageInfo<>(accountDistributeLogVOList);
        return new GenericsResult<>(pageInfoList);
    }
}
