package com.yaoyaozw.customer.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.yaoyaozw.customer.common.BaseResult;
import com.yaoyaozw.customer.common.GenericsResult;
import com.yaoyaozw.customer.components.CustomerServiceCommonAsyncComponent;
import com.yaoyaozw.customer.components.SnowflakeComponent;
import com.yaoyaozw.customer.components.TokenManager;
import com.yaoyaozw.customer.constants.CustomerCommonConstant;
import com.yaoyaozw.customer.dto.customer.CustomerDelayQueryDTO;
import com.yaoyaozw.customer.dto.customer.CustomerDelaySaveDTO;
import com.yaoyaozw.customer.entity.CommonReferralBody;
import com.yaoyaozw.customer.entity.AuthorizerInfo;
import com.yaoyaozw.customer.entity.CustomerGraphicsDelay;
import com.yaoyaozw.customer.entity.ReferralEntity;
import com.yaoyaozw.customer.mapper.CustomerGraphicsDelayMapper;
import com.yaoyaozw.customer.mapper.KanbanCommonMapper;
import com.yaoyaozw.customer.mapper.MaterialCommonMapper;
import com.yaoyaozw.customer.service.AuthorizerInfoService;
import com.yaoyaozw.customer.service.CustomerGraphicsDelayService;
import com.yaoyaozw.customer.service.ReferralEntityService;
import com.yaoyaozw.customer.vo.AuthInfoVO;
import com.yaoyaozw.customer.vo.kanban.CommonOptionResponseVO;
import com.yaoyaozw.customer.vo.customer.CustomerDelayGraphicsDetailVO;
import com.yaoyaozw.customer.vo.customer.CustomerDelayItemVO;
import com.yaoyaozw.customer.vo.customer.CustomerDelayListVO;
import jodd.util.StringUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author darker
 * @date 2022/9/15 11:13
 */
@Service
public class CustomerGraphicsDelayServiceImpl extends ServiceImpl<CustomerGraphicsDelayMapper, CustomerGraphicsDelay> implements CustomerGraphicsDelayService {

    private final static Logger localLog = LoggerFactory.getLogger(CustomerGraphicsDelayServiceImpl.class);

    private final static String EMPTY_STR = " ";

    @Autowired
    private TokenManager tokenManager;
    @Autowired
    private SnowflakeComponent snowflakeComponent;
    @Autowired
    private ReferralEntityService referralEntityService;
    @Autowired
    private AuthorizerInfoService authorizerInfoService;
    @Autowired
    private KanbanCommonMapper kanbanCommonMapper;
    @Autowired
    private CustomerServiceCommonAsyncComponent commonAsyncComponent;
    @Autowired
    private MaterialCommonMapper materialCommonMapper;


    @Override
    public List<CustomerDelayItemVO> findAllDelayCustomerSort(Set<String> appidList,Integer maxSortRange) {

       return baseMapper.findDelayCustomerMessageByRange(appidList,maxSortRange);

    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateSendNumById(Set<CustomerDelayItemVO> voList) {

        baseMapper.updateSendNumByVo(voList);
    }


    @Override
    public BaseResult insertCustomerDelay(CustomerDelaySaveDTO saveDto) {
        StringBuilder builder = new StringBuilder();
        if (StringUtils.isBlank(saveDto.getName())) {
            builder.append("请填写标题");
        }
        if (ObjectUtil.isNull(saveDto.getTimeInterval())) {
            builder.append(StringUtils.isBlank(builder) ? "请填写时间" : "、时间");
        }
        if (StringUtils.isNotBlank(builder)) {
            return new GenericsResult<>(false, builder.toString());
        }

        CustomerGraphicsDelay customerGraphicsDelay = new CustomerGraphicsDelay();
        BeanUtil.copyProperties(saveDto, customerGraphicsDelay);
        // 初始化操作信息
        customerGraphicsDelay.initOperateInfo(tokenManager.getUserIdFromToken(), ObjectUtil.isNull(saveDto.getId()));
        if (ObjectUtil.isNull(customerGraphicsDelay.getId())) {
            long id = snowflakeComponent.snowflakeId();
            localLog.info("是新增, 新生成ID: {}", id);
            customerGraphicsDelay.setId(id);
        }

        localLog.info("生成referral数据");
        ReferralEntity referralEntity = new ReferralEntity();
        BeanUtil.copyProperties(saveDto.getCustomerReferralDto(), referralEntity);
        referralEntity.setMaterialGraphicsId(customerGraphicsDelay.getId());

        // 非自定义获取链接
        if (!referralEntity.getNewsType().equals(-1)) {
            try {
                customerGraphicsDelay.setSendStatus(CustomerCommonConstant.SEND_STATUS_LINK_GETTING);
                localLog.info("准备获取链接");
                String linkNameModel = CustomerCommonConstant.getLinkNameModel(referralEntity.getNewsType());
                if (StringUtils.isNotBlank(linkNameModel)) {
                    localLog.info("获取链接name模板: {}", linkNameModel);
                    referralEntity.setName(linkNameModel);
                }
                String storeTypeName = materialCommonMapper.getStoreTypeNameByAppId(customerGraphicsDelay.getAppId());
                referralEntity.setStoreTypeName(storeTypeName);
                SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd");
                String dateStr = dateFormat.format(new Date());

                AuthorizerInfo authorizerInfo = authorizerInfoService.getOne(new QueryWrapper<AuthorizerInfo>().eq("appid", saveDto.getAppId()));
                AuthInfoVO authInfoVO = new AuthInfoVO();
                authInfoVO.putPropertyValue(authorizerInfo);
                commonAsyncComponent.dispatchDelay(dateStr, authInfoVO, referralEntity);
                customerGraphicsDelay.setSendStatus(CustomerCommonConstant.SEND_STATUS_ACTIVE);
                localLog.info("链接获取完成");
            } catch (Exception e) {
                localLog.info("获取链接异常");
                return new BaseResult().error("获取链接异常" + e.getMessage());
            }
        }
        try {
            customerGraphicsDelay.setSourceUrl(referralEntity.getReferral());
        } catch (Exception e) {
            return new BaseResult().error(e.getMessage());
        }

        boolean mainResult = super.saveOrUpdate(customerGraphicsDelay);
        if (!mainResult) {
            localLog.info("主体数据生成异常");
            return new BaseResult().error("新增/更新 主体数据异常");
        }

        boolean referralResult = referralEntityService.saveOrUpdate(referralEntity);
        if (!referralResult) {
            return new BaseResult().error("新增/更新 链接数据异常");
        }
        reSortCustomer(saveDto.getAppId());

        return new BaseResult().success();
    }

    @Override
    public GenericsResult<CustomerDelayGraphicsDetailVO> getCustomerDelayDetail(Long id) {
        CustomerGraphicsDelay customerGraphicsDelay = super.getById(id);
        if (ObjectUtil.isNull(customerGraphicsDelay)) {
            return new GenericsResult<>(false, "找不到主体数据");
        }
        CustomerDelayGraphicsDetailVO customerDelayGraphicsDetailVO = new CustomerDelayGraphicsDetailVO();
        BeanUtil.copyProperties(customerGraphicsDelay, customerDelayGraphicsDetailVO);

        // 查询链接数据
        ReferralEntity referralEntity = referralEntityService.getOne(new QueryWrapper<ReferralEntity>().eq("material_graphics_id", id));

        if (ObjectUtil.isNotNull(referralEntity)) {
            CommonReferralBody customerReferralDto = new CommonReferralBody();
            BeanUtil.copyProperties(referralEntity, customerReferralDto);

            customerDelayGraphicsDetailVO.setCustomerReferralDto(customerReferralDto);
        } else {
            customerDelayGraphicsDetailVO.setCustomerReferralDto(new CommonReferralBody());
        }

        AuthInfoVO authInfoVO = super.baseMapper.getCustomerDelayAuthInfo(id);
        customerDelayGraphicsDetailVO.setAuthInfoVo(authInfoVO);

        return new GenericsResult<>(customerDelayGraphicsDetailVO);
    }

    @Override
    public GenericsResult<PageInfo<CustomerDelayListVO>> pageList(CustomerDelayQueryDTO queryDto) {
        PageHelper.startPage(queryDto.getCurrentPage(), queryDto.getPageSize());
        List<CustomerDelayListVO> customerDelayList = super.baseMapper.pageList(queryDto);
        PageInfo<CustomerDelayListVO> pageResult = new PageInfo<>(customerDelayList);

        return new GenericsResult<>(pageResult);
    }

    @Override
    public GenericsResult<List<AuthInfoVO>> getExistCustomerAccount(String keyword) {
        List<AuthInfoVO> existCustomerAccount = super.baseMapper.getExistCustomerAccount(keyword);
        return new GenericsResult<>(existCustomerAccount);
    }

    @Override
    public BaseResult removeCustomerDelay(Long id) {
        CustomerGraphicsDelay entity = super.getById(id);
        boolean result = super.removeById(id);
        if (result) {
            referralEntityService.ultimateDeleteReferrals(id, null);

            localLog.info("处理完成, 正在重新排序");
            reSortCustomer(entity.getAppId());
            return new BaseResult().success();
        }
        return new BaseResult().error("弃用失败");
    }

    @Override
    public BaseResult removeBatch(List<Long> idList) {
        List<CustomerGraphicsDelay> list = (List<CustomerGraphicsDelay>) super.listByIds(idList);
        boolean result = super.removeByIds(idList);
        if (result) {
            referralEntityService.ultimateDeleteReferralsBatch(idList, null);

            localLog.info("处理完成, 正在重新排序");
            Set<String> appIdSet = list.stream().map(CustomerGraphicsDelay::getAppId).collect(Collectors.toSet());
            for (String appId : appIdSet) {
                reSortCustomer(appId);
            }
            return new BaseResult().success();
        }
        return new BaseResult().error("删除失败");
    }

    @Override
    public BaseResult copy(String appId, String bookId, String bookName, Integer chapterIdx, List<String> targetAppList) {
        List<AuthorizerInfo> authorizerInfoList = authorizerInfoService.list();
        Map<String, AuthorizerInfo> authMap = authorizerInfoList.stream().collect(Collectors.toMap(AuthorizerInfo::getAppid, Function.identity()));

        AuthorizerInfo sourceAuthInfo = authMap.get(appId);
        if (ObjectUtil.isNull(sourceAuthInfo)) {
            return new BaseResult().error("找不到公众号信息");
        }
        String accountId = sourceAuthInfo.getAccountId();
        localLog.info("原始accountId: {}", accountId);

        // 找到号下面的所有的客服素材
        List<CustomerGraphicsDelay> sourceCustomerList = super.list(new QueryWrapper<CustomerGraphicsDelay>().eq("app_id", appId));
        List<Long> sourceMainIdList = sourceCustomerList.stream().map(CustomerGraphicsDelay::getId).collect(Collectors.toList());
        // 获取原始链接素材 图文只有一条、文本有多条
        List<ReferralEntity> referralEntityList = referralEntityService.list(new QueryWrapper<ReferralEntity>().in("material_graphics_id", sourceMainIdList).orderByAsc("gmt_create").orderByAsc("sort"));
        // 分组链接，如果复用时选择了书，则需要为原本的所有书进行重新赋值
        Stream<ReferralEntity> stream = referralEntityList.stream();
        if (StringUtil.isNotBlank(bookId)) {
            stream = stream.peek(item -> {
                // 选了书，将推广类型的链接书籍换成选择的书籍
                if (StringUtil.isNotBlank(item.getBookId()) && CustomerCommonConstant.BOOK_NEWS_TYPE.equals(item.getNewsType())) {
                    item.setBookId(bookId);
                    item.setBookName(bookName);
                    if (ObjectUtil.isNotNull(chapterIdx)) {
                        item.setChapterIdx(chapterIdx);
                    }
                }
            });
        }
        Map<Long, List<ReferralEntity>> referralGroupMap = stream.collect(Collectors.groupingBy(ReferralEntity::getMaterialGraphicsId));

        // 遍历处理
        HashSet<String> storeTypeKeySet = CollectionUtil.newHashSet(sourceAuthInfo.getStoreType());
        List<CommonOptionResponseVO> storeTypeList = kanbanCommonMapper.getStoreTypeEntity(storeTypeKeySet);
        // 这里只有一个
        String storeTypeName = storeTypeList.get(0).getName();
        localLog.info("当前复用书城: {}", storeTypeName);
        sourceAuthInfo.setStoreTypeName(storeTypeName);

        // 遍历要复用的素材
        for (CustomerGraphicsDelay item : sourceCustomerList) {
            // 设置所属链接
            item.setBelongReferralList(referralGroupMap.get(item.getId()));
            // 为公众号设置书城名
            item.setAuthorizerInfo(sourceAuthInfo);
        }

        for (String targetAppId : targetAppList) {
            // 构造目标公众号的信息
            AuthorizerInfo targetAuthInfo = authMap.get(targetAppId);
            targetAuthInfo.setStoreTypeName(storeTypeName);
            // 异步复用
            commonAsyncComponent.obtainDelayGraphicsLink(sourceCustomerList, targetAuthInfo);
        }
        return new BaseResult().success("复用进行中, 请等待");
    }

    @Override
    public GenericsResult<List<AuthInfoVO>> getAuthList(String keyword, String appId) {
        String storeType = null;
        if (ObjectUtil.isNotNull(appId)) {
            AuthorizerInfo authInfo = authorizerInfoService.getOne(new QueryWrapper<AuthorizerInfo>().eq("appid", appId));
            storeType = authInfo.getStoreType();
        }
        List<String> keywordList = null;
        // 拆分多关键词
        if (ObjectUtil.isNotNull(keyword) && keyword.contains(EMPTY_STR)) {
            String[] keywordSplit = keyword.split(EMPTY_STR);
            keywordList = Arrays.asList(keywordSplit);
            keyword = null;
        }

        List<AuthInfoVO> authList = super.baseMapper.getAuthList(keyword, keywordList, storeType, appId);
        return new GenericsResult<>(authList);
    }

    @Override
    public GenericsResult<List<CommonOptionResponseVO>> getTimeIntervalList() {
        List<CommonOptionResponseVO> timeIntervalList = super.baseMapper.getTimeIntervalList();
        return new GenericsResult<>(timeIntervalList);
    }

    @Override
    public void reSortCustomer(String appId) {
        List<CustomerGraphicsDelay> list = super.list(new QueryWrapper<CustomerGraphicsDelay>().eq("app_id", appId).orderByAsc("time_interval"));
        int idx = 0;
        Long previousTime = 0L;
        for (CustomerGraphicsDelay item : list) {
            boolean equal = previousTime.equals(item.getTimeInterval());

            if (!equal) {
                idx += 1;
            }
            item.setPostSort(idx);
            previousTime = item.getTimeInterval();
        }
        if (CollectionUtil.isNotEmpty(list)) {
            super.updateBatchById(list);
        }

    }


}

