package com.yaoyaozw.customer.aop;

import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.yaoyaozw.customer.annotations.AccountOperateControl;
import com.yaoyaozw.customer.common.BaseResult;
import com.yaoyaozw.customer.components.TokenManager;
import com.yaoyaozw.customer.constants.CustomerCommonConstant;
import com.yaoyaozw.customer.entity.AccountSetup;
import com.yaoyaozw.customer.enums.AccountParamType;
import com.yaoyaozw.customer.enums.AccountTableColumnType;
import com.yaoyaozw.customer.exception.AopCommonException;
import com.yaoyaozw.customer.mapper.MaterialAopCommonMapper;
import com.yaoyaozw.customer.service.AccountSetupService;
import org.apache.commons.lang3.ObjectUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.lang.reflect.Field;
import java.lang.reflect.Method;

/**
 * @author darker
 * @date 2022/12/13 12:11
 */
@Component
@Aspect
public class AccountOperateControlAop {

    private final static Logger localLog = LoggerFactory.getLogger(AccountOperateControlAop.class);


    @Autowired
    private TokenManager tokenManager;
    @Autowired
    private AccountSetupService accountSetupService;
    @Autowired
    private MaterialAopCommonMapper aopCommonMapper;
/*
    @Pointcut("@annotation(com.yaoyaozw.customer.annotations.AccountOperateControl)")
    public void operateControlCut() {
        //公众号权限控制切点
    }

    @Before("operateControlCut()")
    public void operateControl(JoinPoint joinPoint) throws Throwable {
        // 获取注解的详细信息
        MethodSignature methodSignature = (MethodSignature) joinPoint.getSignature();
        Method declaredMethod = methodSignature.getMethod();
        AccountOperateControl operateControl = declaredMethod.getAnnotation(AccountOperateControl.class);

        // 获取校验字段值
        String paramName = operateControl.paramName();
        int index = 0;
        for (String parameterName : methodSignature.getParameterNames()) {
            if (parameterName.equals(paramName)) {
                break;
            }
            index++;
        }
        Object paramObj = joinPoint.getArgs()[index];
        if (ObjectUtil.isNull(paramObj)) {
            throw new AopCommonException("找不到校验目标");
        }

        // 解析注解的内容
        AccountParamType type = operateControl.paramType();
        BaseResult baseResult;
        // 分类校验
        Long userIdLong = tokenManager.getUserIdFromToken();
        String username = tokenManager.getUserNameFromToken();
        String userId = userIdLong.toString();
        localLog.info("操作人Id: {}, 用户名: {}", userId, username);
        if (AccountParamType.TABLE_PRIMARY.equals(type)) {
            // 是以表的主键为参数的
            baseResult = handleTablePrimary(userId, paramObj.toString(), operateControl);
        } else if (AccountParamType.ACCOUNT_PRIMARY.equals(type)) {
            // 是以公众号id为参数的
            baseResult = handleAccountPrimary(userId, paramObj.toString(), operateControl);
        } else if (AccountParamType.REQUEST_BODY.equals(type)) {
            // 是以包含公众号字段的实体为参数的
            baseResult = handleRequestBody(userId, paramObj, operateControl);
        } else {
            baseResult = new BaseResult().error("权限校验异常-未知类型");
        }
        if (!baseResult.getSuccess()) {
            throw new AopCommonException(baseResult.getMessage());
        }

    }*/

    /**
     * 校验以表主键id为参数的请求权限
     * @param userId 用户名
     * @param primaryId 表主键id
     * @param annotation 注解
     * @return 校验结果
     */
    private BaseResult handleTablePrimary(String userId, String primaryId, AccountOperateControl annotation) {
        if (!ObjectUtils.allNotNull(userId, primaryId, annotation.table(), annotation.column())) {
            return new BaseResult().error("校验参数缺失");
        }
        // 通过参数获取这条数据对应的公众号的id
        String accountParam = aopCommonMapper.getAccountFromTable(primaryId, annotation.table(), annotation.column());
        localLog.info("TABLE_PRIMARY 类型 获取公众号ID: {}", accountParam);
        // 调用方法处理username和公众号id的匹配关系

        Boolean result = checkUserAccount(userId, accountParam, annotation.columnType());
        return result ? new BaseResult().success() : new BaseResult().error("没有该公众号的操作权限");
    }

    /**
     * 校验以公众号主键id为参数的请求
     * @param userId 用户名
     * @param accountParam 公众号主键id
     * @return 检验结果
     */
    private BaseResult handleAccountPrimary(String userId, String accountParam, AccountOperateControl annotation) {
        localLog.info("ACCOUNT_PRIMARY 类型 获取公众号ID: {}", accountParam);
        Boolean result = checkUserAccount(userId, accountParam,annotation.columnType());

        return result ? new BaseResult().success() : new BaseResult().error("没有该公众号的操作权限");
    }

    /**
     * 校验以实体类为参数的请求
     * @param userId 用户名
     * @param requestBody 实体类
     * @param annotation 注解
     * @return 校验结果
     */
    private BaseResult handleRequestBody(String userId, Object requestBody, AccountOperateControl annotation) throws Exception {
        // 获取实体中的公众号id字段值
        Class<?> clazz = requestBody.getClass();
        Field field = clazz.getDeclaredField(annotation.property());
        field.setAccessible(true);
        Object obj = field.get(requestBody);
        if (ObjectUtil.isNull(obj)) {
            return new BaseResult().error("无法获取公众号校验字段");
        }
        String accountParam = obj.toString();
        localLog.info("REQUEST_BODY 类型 获取公众号ID: {}", accountParam);
        Boolean result = checkUserAccount(userId, accountParam, annotation.columnType());
        return result ? new BaseResult().success() : new BaseResult().error("没有该公众号的操作权限");
    }

    /**
     * 最终校验
     * @param userId 用户id
     * @param accountParam 公众号相关参数（authId / appId）
     * @param columnType 字段类型（authId / appId）
     * @return 校验结果(true-通过；false-未通过)
     */
    private Boolean checkUserAccount(String userId, String accountParam, AccountTableColumnType columnType) {
        // 获取用户角色等级
        Integer userRoleLevel = aopCommonMapper.getUserRoleLevel(userId);
        if (userRoleLevel >= CustomerCommonConstant.PERMISSION_OPENING_ROLE_LEVEL) {
            // 负责人及以上级别不加控制
            return Boolean.TRUE;
        }
        QueryWrapper<AccountSetup> queryWrapper = new QueryWrapper<AccountSetup>().eq(AccountSetup.COL_USER_ID, userId);
        if (AccountTableColumnType.AUTH_ID.equals(columnType)) {
            queryWrapper = queryWrapper.eq(AccountSetup.COL_AUTH_ID, accountParam);
        } else if (AccountTableColumnType.APP_ID.equals(columnType)) {
            queryWrapper = queryWrapper.eq(AccountSetup.COL_APP_ID, accountParam);
        }
        int count = accountSetupService.count(queryWrapper);
        return count != 0;
    }

}
