package com.yaoyaozw.customer.controller;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.google.common.collect.Lists;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.yaoyaozw.customer.components.CustomerServiceCommonAsyncComponent;
import com.yaoyaozw.customer.constants.CustomerMaterialConstant;
import com.yaoyaozw.customer.dto.integration.IntegrationRequestDTO;
import com.yaoyaozw.customer.entity.CustomerFollowReply;
import com.yaoyaozw.customer.entity.CustomerGraphics;
import com.yaoyaozw.customer.entity.CrowdPackage;
import com.yaoyaozw.customer.entity.RegisterUserEntity;
import com.yaoyaozw.customer.mapper.KanbanCommonMapper;
import com.yaoyaozw.customer.publisher.AcquisitionExceptionEventPublisher;
import com.yaoyaozw.customer.service.*;
import com.yaoyaozw.customer.service.wechat.service.WeChatService;
import com.yaoyaozw.customer.vo.AuthInfoVO;
import com.yaoyaozw.customer.vo.customer.CrowdPackageUserVO;
import com.yaoyaozw.customer.vo.referral.ReferralEntityVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * @author wgh
 * @date 2022/10/10 20:04
 */
@Slf4j
@RestController
@RequestMapping("/customer-service/test")
public class TestController {

    private final static String STORE_ENTITY_REDIS_KEY = "STORE_NAME_MAP";

    @Autowired
    private CustomerGraphicsService customerGraphicsService;
    @Autowired
    private AccountOrderService accountOrderService;
    @Autowired
    private CustomerDelayPublishService customerDelayPublishService;
    @Autowired
    private RabbitTemplate rabbitTemplate;
    @Autowired
    private RedisTemplate<String, Object> redisTemplate;
    @Autowired
    private KanbanCommonMapper kanbanCommonMapper;

    @GetMapping("/CustomerTest")
    public void test(){

        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.AM_PM,0);
        calendar.set(Calendar.HOUR,16);
        calendar.set(Calendar.MINUTE,0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);

        IntegrationRequestDTO integrationRequestDTO = new IntegrationRequestDTO();
        integrationRequestDTO.setRequestDate(calendar.getTime());
        customerGraphicsService.sendCustomerMessage(integrationRequestDTO);
    }

    @GetMapping("/CustomerDelayTest")
    public void delayTest(){
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.AM_PM,0);
        calendar.set(Calendar.HOUR,17);
        calendar.set(Calendar.MINUTE,0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);

        IntegrationRequestDTO integrationRequestDTO = new IntegrationRequestDTO();
        integrationRequestDTO.setRequestDate(calendar.getTime());
        customerDelayPublishService.sendCustomerDelayMessage(integrationRequestDTO);
    }

    @GetMapping("/payTest")
    public void payTest(){

        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.AM_PM,0);
        calendar.set(Calendar.HOUR,12);
        calendar.set(Calendar.MINUTE,0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);

        IntegrationRequestDTO integrationRequestDTO = new IntegrationRequestDTO();
        integrationRequestDTO.setRequestDate(calendar.getTime());
        accountOrderService.activeUserByOrder(integrationRequestDTO);
    }

    @GetMapping("/putRedisStore")
    public void redisStore() {
        List<AuthInfoVO> allStoreList = kanbanCommonMapper.getAllStoreList();
        Map<String, String> storeMap = allStoreList.stream().collect(Collectors.toMap(AuthInfoVO::getStoreType, AuthInfoVO::getStoreTypeName));
        redisTemplate.opsForHash().putAll(STORE_ENTITY_REDIS_KEY, storeMap);

    }

    @GetMapping("/testFollowReply")
    public void testFollowReply() {
        Object obj = redisTemplate.opsForHash().get(CustomerMaterialConstant.FOLLOW_REPLY_REDIS_KEY, "wx9f460fa46420016c");
        if (ObjectUtil.isNotNull(obj)) {

            List<CustomerFollowReply> list = JSONUtil.toList(JSONUtil.parseArray(obj.toString()), CustomerFollowReply.class);
            for (CustomerFollowReply customerFollowReply : list) {
                System.out.println(customerFollowReply);
            }
        }
    }


    @Resource
    private WeChatService weChatService;
    @Resource
    private RegisterUserEntityService registerUserEntityService;
    @Resource
    private ReferralEntityService referralEntityService;

    @GetMapping("/testSendCustomer")
    public void testSendCustomer() {
        CustomerGraphics entity = customerGraphicsService.getById(1874011792770990080L);

        List<CrowdPackageUserVO> userList = registerUserEntityService.getCurrentInPackUserList(entity.getPackId(), false);

        List<ReferralEntityVo> referralEntityVos = referralEntityService.findReferralByCustomerGraphicsId(entity.getId());

        weChatService.sendCustomerMessage(userList.get(0).getAppId(), entity, userList, referralEntityVos);
    }

    @Autowired
    private CustomerServiceCommonAsyncComponent asyncComponent;

    @GetMapping("/refreshCrowdPackageUserCount")
    public void refreshCrowdPackageUserCount(@RequestParam Long id) {
        asyncComponent.addMatchUserIntoPackage(id, true);
    }


    @Resource
    private AcquisitionExceptionEventPublisher acquisitionExceptionEventPublisher;

    @GetMapping("/testAcquisitionReplace")
    public void testAcquisitionReplace(@RequestParam String linkId) {
        acquisitionExceptionEventPublisher.inspectAndExecute(linkId);
    }

    @GetMapping("/refreshCustomerMaterialSendNum")
    public void refreshCustomerMaterialSendNum() {
        log.info("开始刷新客服消息送达人数...");

        Calendar instance = Calendar.getInstance();
        instance.add(Calendar.DAY_OF_YEAR, -3);
        Date startTime = instance.getTime();

        List<CustomerGraphics> list = customerGraphicsService.list(new QueryWrapper<CustomerGraphics>().between("post_time", startTime, new Date()));
        log.info("开始刷新客服消息送达人数，待处理客服消息数：{}", list.size());

        Map<Object, Object> customerCount = redisTemplate.opsForHash().entries("CUSTOMER_COUNT");
        Map<Long, Integer> map = new HashMap<>();
        for (Map.Entry<Object, Object> objectObjectEntry : customerCount.entrySet()) {
            map.put(Long.valueOf(objectObjectEntry.getKey().toString()), Integer.valueOf(objectObjectEntry.getValue().toString()));
        }

        List<CustomerGraphics> updateList = new ArrayList<>();
        for (CustomerGraphics customerGraphics : list) {
            CustomerGraphics entity = new CustomerGraphics();
            entity.setId(customerGraphics.getId());
            entity.setSendNum(map.getOrDefault(customerGraphics.getId(), 0));
            updateList.add(entity);
        }
        customerGraphicsService.updateBatchById(updateList);
        log.info("刷新客服消息送达人数完成...");
    }

    @Autowired
    private CrowdPackageService crowdPackageService;

    @GetMapping("/cleanupUserPackages")
    public void cleanupUserPackages() {
        log.info("开始清理用户人群包数据...");
        
        try {
            // 1、查询当前数据库中的人群包数据
            List<CrowdPackage> allPackages = crowdPackageService.list();
            Set<Long> validPackageIds = allPackages.stream().map(CrowdPackage::getId).collect(Collectors.toSet());
            log.info("当前有效人群包数量：{}", validPackageIds.size());
            
            // 2、查询当前数据库中微信用户列表：is_deleted = 0、in_package非空
            QueryWrapper<RegisterUserEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("is_deleted", 0)
                       .isNotNull("in_package")
                       .ne("in_package", "");
            List<RegisterUserEntity> userList = registerUserEntityService.list(queryWrapper);
            log.info("待处理用户数量：{}", userList.size());
            
            if (userList.isEmpty()) {
                log.info("没有需要处理的用户数据");
                return;
            }
            
            // 创建线程池，最多5个线程
            ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
            executor.setCorePoolSize(5);
            executor.setMaxPoolSize(5);
            executor.setQueueCapacity(100);
            executor.setKeepAliveSeconds(60);
            executor.setThreadNamePrefix("PackageCleanup-");
            executor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
            executor.initialize();
            
            // 将用户列表分组，使用Lists.partition方法分组
            int batchSize = Math.max(1, userList.size() / 5);
            List<List<RegisterUserEntity>> userBatches = Lists.partition(userList, batchSize);
            
            // 用于收集所有处理结果的并发安全Map
            ConcurrentHashMap<String, List<Long>> groupedResults = new ConcurrentHashMap<>();
            
            // 提交任务到线程池
            List<Future<Void>> futures = new ArrayList<>();
            for (List<RegisterUserEntity> batch : userBatches) {
                Future<Void> future = executor.submit(() -> {
                    log.info("线程 {} 开始处理 {} 个用户", Thread.currentThread().getName(), batch.size());
                    
                    for (RegisterUserEntity user : batch) {
                        try {
                            // 3、转换每个用户的in_package值为Long列表
                            String inPackage = user.getInPackage();
                            if (StrUtil.isBlank(inPackage)) {
                                continue;
                            }
                            
                            List<Long> packageIds = Arrays.stream(inPackage.split(","))
                                .filter(StrUtil::isNotEmpty)
                                .map(String::trim)
                                .filter(s -> s.matches("\\d+"))
                                .map(Long::valueOf)
                                .collect(Collectors.toList());
                            
                            // 4、移除不在有效人群包列表中的值
                            List<Long> validIds = packageIds.stream()
                                .filter(validPackageIds::contains)
                                .sorted()
                                .collect(Collectors.toList());
                            
                            // 5、生成新的in_package字符串
                            String newInPackage;
                            if (validIds.isEmpty()) {
                                newInPackage = "";
                            } else {
                                newInPackage = validIds.stream()
                                    .map(String::valueOf)
                                    .collect(Collectors.joining(","));
                            }
                            
                            // 6、按照in_package分组
                            synchronized (groupedResults) {
                                groupedResults.computeIfAbsent(newInPackage, k -> new ArrayList<>()).add(user.getId());
                            }
                            
                        } catch (Exception e) {
                            log.error("处理用户 {} 时发生错误: {}", user.getId(), e.getMessage(), e);
                        }
                    }
                    
                    log.info("线程 {} 处理完成", Thread.currentThread().getName());
                    return null;
                });
                futures.add(future);
            }
            
            // 等待所有任务完成
            for (Future<Void> future : futures) {
                try {
                    future.get();
                } catch (Exception e) {
                    log.error("等待线程执行完成时发生错误: {}", e.getMessage(), e);
                }
            }
            
            executor.shutdown();
            
            // 7、遍历Map，批量更新user_register_entity的in_package字段
            log.info("开始批量更新数据库，共 {} 个分组", groupedResults.size());
            
            for (Map.Entry<String, List<Long>> entry : groupedResults.entrySet()) {
                String newInPackage = entry.getKey();
                List<Long> userIds = entry.getValue();
                
                try {
                    // 分批更新，避免单次更新数据过多
                    int updateBatchSize = 1000;
                    List<List<Long>> batchIdsList = Lists.partition(userIds, updateBatchSize);
                    
                    for (List<Long> batchIds : batchIdsList) {
                        // 构造更新实体
                        RegisterUserEntity updateEntity = new RegisterUserEntity();
                        updateEntity.setInPackage(newInPackage);
                        
                        // 构造查询条件：id in (batchIds)
                        QueryWrapper<RegisterUserEntity> updateWrapper = new QueryWrapper<>();
                        updateWrapper.in("id", batchIds);
                        
                        registerUserEntityService.update(updateEntity, updateWrapper);
                        log.info("更新了 {} 个用户的in_package为: '{}'", batchIds.size(), newInPackage);
                        TimeUnit.MILLISECONDS.sleep(500);
                    }
                } catch (Exception e) {
                    log.error("批量更新in_package为 '{}' 的用户时发生错误: {}", newInPackage, e.getMessage(), e);
                }
            }
            
            log.info("用户人群包数据清理完成！");
            
        } catch (Exception e) {
            log.error("清理用户人群包数据时发生错误: {}", e.getMessage(), e);
        }
    }

}
