package com.yaoyaozw.customer.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;

import cn.hutool.core.util.ObjectUtil;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.yaoyaozw.customer.common.BaseResult;
import com.yaoyaozw.customer.common.GenericsResult;
import com.yaoyaozw.customer.components.CustomerServiceCommonAsyncComponent;
import com.yaoyaozw.customer.components.SnowflakeComponent;
import com.yaoyaozw.customer.components.TokenManager;
import com.yaoyaozw.customer.constants.CrowdPackageCommonConstant;
import com.yaoyaozw.customer.constants.CustomerCommonConstant;
import com.yaoyaozw.customer.dto.customer.CustomerBatchSetPackDTO;
import com.yaoyaozw.customer.dto.customer.CustomerMessageQueryDTO;
import com.yaoyaozw.customer.dto.customer.CustomerMessageSaveDTO;
import com.yaoyaozw.customer.dto.integration.IntegrationRequestDTO;
import com.yaoyaozw.customer.entity.*;
import com.yaoyaozw.customer.mapper.KanbanCommonMapper;
import com.yaoyaozw.customer.service.CrowdPackageConditionMatchService;
import com.yaoyaozw.customer.service.CrowdPackageConditionService;
import com.yaoyaozw.customer.service.ReferralEntityService;
import com.yaoyaozw.customer.vo.customer.*;
import com.yaoyaozw.customer.service.*;
import com.yaoyaozw.customer.service.wechat.service.WeChatService;
import com.yaoyaozw.customer.vo.customer.CrowdPackageUserVO;
import org.apache.commons.lang3.StringUtils;
import com.yaoyaozw.customer.vo.referral.ReferralEntityVo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import java.util.List;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yaoyaozw.customer.mapper.CustomerGraphicsMapper;
import com.yaoyaozw.customer.service.CustomerGraphicsService;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import static java.util.stream.Collectors.toCollection;

/**
 * 客户图形服务impl
 *
 * @author Admin
 * @date 2022/09/28
 */
@Service
public class CustomerGraphicsServiceImpl extends ServiceImpl<CustomerGraphicsMapper, CustomerGraphics> implements CustomerGraphicsService {

    private final static Logger LOCAL_LOG = LoggerFactory.getLogger(CustomerGraphicsServiceImpl.class);


    @Autowired
    private CrowdPackageConditionMatchService matchService;
    @Autowired
    private TokenManager tokenManager;
    @Autowired
    private CustomerServiceCommonAsyncComponent commonAsyncComponent;
    @Autowired
    private ReferralEntityService referralEntityService;
    @Autowired
    private WeChatService weChatService;
    @Autowired
    private RegisterUserEntityService registerUserEntityService;

    @Autowired
    private CrowdPackageConditionService conditionService;
    @Autowired
    private SnowflakeComponent snowflakeComponent;
    @Autowired
    private KanbanCommonMapper kanbanCommonMapper;

    @Autowired
    private CrowdPackageService crowdPackageService;


    @Override
    public BaseResult insertCustomerMessage(CustomerMessageSaveDTO saveDto) {

        // 主体数据
        LOCAL_LOG.info("处理主体数据");
        CustomerGraphics customerGraphics = new CustomerGraphics();
        BeanUtil.copyProperties(saveDto, customerGraphics);
        customerGraphics.initOperateInfo(tokenManager.getUserIdFromToken(), ObjectUtil.isNull(saveDto.getId()));
        if (ObjectUtil.isNull(customerGraphics.getId())) {
            long id = snowflakeComponent.snowflakeId();
            customerGraphics.setId(id);
        }
        // 设置链接数
        if (CustomerCommonConstant.REMOTE_LINK_NEWS_TYPE_LIST.contains(saveDto.getCustomerReferralDto().getNewsType())) {
            customerGraphics.setReferralSize(1);
        }
        super.saveOrUpdate(customerGraphics);

        // 处理活动数据
        ReferralEntity referralEntity = new ReferralEntity();
        BeanUtil.copyProperties(saveDto.getCustomerReferralDto(), referralEntity);
        referralEntity.setMaterialGraphicsId(customerGraphics.getId());
        // 获取name模板
        String nameModel = CustomerCommonConstant.getLinkNameModel(referralEntity.getNewsType());
        LOCAL_LOG.info("获取name模板: {}", nameModel);
        if (StringUtils.isNotBlank(nameModel)) {
            referralEntity.setName(nameModel);
        }

        // 保存链接数据
        referralEntityService.saveOrUpdate(referralEntity);
        return new BaseResult().success();
    }

    @Override
    public GenericsResult<CustomerMessageDetailVO> getCustomerMessageDetail(Long id) {

        // 获取主体数据
        CustomerGraphics customerGraphics = super.getById(id);
        if (ObjectUtil.isNull(customerGraphics)) {
            return new GenericsResult<>(false, "找不到主体数据");
        }
        CustomerMessageDetailVO customerMessageDetailVO = new CustomerMessageDetailVO();
        BeanUtil.copyProperties(customerGraphics, customerMessageDetailVO);
        customerMessageDetailVO.setPostTime(customerGraphics.getPostTimeStr());

        // 获取链接数据
        ReferralEntity referralEntity = referralEntityService.getOne(new QueryWrapper<ReferralEntity>().eq("material_graphics_id", id).isNull("account_id"));
        CommonReferralBody customerReferralDto = new CommonReferralBody();

        if (referralEntity != null) {
            BeanUtil.copyProperties(referralEntity, customerReferralDto);
        }
        customerMessageDetailVO.setCustomerReferralDto(customerReferralDto);

        return new GenericsResult<>(customerMessageDetailVO);
    }

    @Override
    public GenericsResult<PageInfo<CustomerMessageListVO>> pageList(CustomerMessageQueryDTO queryDto) {
        PageHelper.startPage(queryDto.getCurrentPage(), queryDto.getPageSize());
        List<CustomerMessageListVO> list = baseMapper.pageList(queryDto);
        PageInfo<CustomerMessageListVO> pageList = new PageInfo<>(list);
        return new GenericsResult<>(pageList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public BaseResult removeCustomerMessage(Long id) {
        boolean result = super.removeById(id);
        QueryWrapper<ReferralEntity> queryWrapper = new QueryWrapper<ReferralEntity>().eq("material_graphics_id", id);
        boolean referralResult = referralEntityService.remove(queryWrapper);
        if (result && referralResult) {
            // 删除成功
            return new BaseResult().success();
        }
        if (!referralResult) {
            int count = referralEntityService.count(queryWrapper);
            if (count == 0) {
                // 没找到链接数据
                LOCAL_LOG.info("没找到链接数据, 删除完成");
                return new BaseResult().success();
            }
        }
        LOCAL_LOG.info("主体删除结果: {}, 链接删除结果: {}, 删除失败，回滚事务", result, referralResult);
        TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
        return new BaseResult().error("删除异常");
    }

    @Override
    public BaseResult setPack(Long id, Long packId) {
        CustomerGraphics byId = super.getById(id);
        if (ObjectUtil.isNull(byId)) {
            return new BaseResult().error("无法获取主体数据");
        }
        byId.setPackId(packId);

        LOCAL_LOG.info("获取书城条件主体");
        CrowdPackageCondition storeCondition = conditionService.getOne(
                new QueryWrapper<CrowdPackageCondition>().eq("condition_key", CrowdPackageCommonConstant.CONDITION_STORE_ID).eq("front_type", CrowdPackageCommonConstant.MULTIPLE_SELECT)
        );
        //  校验人群包设置条件
        LOCAL_LOG.info("获取人群包下的书城条件");
        CrowdPackageConditionMatch storeConditionMatch = matchService.getOne(
                new QueryWrapper<CrowdPackageConditionMatch>().eq("package_id", packId).eq("condition_id", storeCondition.getId())
        );
        List<ReferralEntity> referralEntityList = referralEntityService.list(new QueryWrapper<ReferralEntity>().eq("material_graphics_id", id).isNull("account_id"));

        for (ReferralEntity referralEntity : referralEntityList) {
            Integer newsType = referralEntity.getNewsType();
            if (CustomerCommonConstant.BOOK_NEWS_TYPE.equals(newsType)) {
                // 是推广链接
                BaseResult checkResult = this.checkExtendBook(referralEntity.getStoreType(), storeConditionMatch);
                if (!checkResult.getSuccess()) {
                    return checkResult;
                }
            }
        }

        byId.setPackId(packId);
        // 设置状态为链接生成中
        byId.setSendStatus(CustomerCommonConstant.SEND_STATUS_LINK_GETTING);
        boolean result = super.updateById(byId);
        if (result) {
            commonAsyncComponent.obtainMessageLink(id, byId, referralEntityList);
            return new BaseResult().success();
        }
        return new BaseResult().error("更新失败");
    }

    @Override
    @Async("myExecutor")
    public void batchSetPack(List<CustomerBatchSetPackDTO> batchSetPackList) {
        if (CollectionUtil.isEmpty(batchSetPackList)) {
            LOCAL_LOG.error("批量设置参数不能为空");
            return;
        }
        LOCAL_LOG.info("开始批量设置人群包，总数量: {}", batchSetPackList.size());
        
        // 收集所有客服消息ID
        List<Long> customerMessageIds = batchSetPackList.stream()
                .map(CustomerBatchSetPackDTO::getCustomerMessageId)
                .collect(Collectors.toList());

        // 先将所有相关客服消息状态设置为批量生成中（状态值：11）
        CustomerGraphics updateEntity = new CustomerGraphics();
        updateEntity.setSendStatus(11);
        this.update(updateEntity, new UpdateWrapper<CustomerGraphics>().in("id", customerMessageIds));
        LOCAL_LOG.info("已将 {} 条客服消息状态设置为批量生成中", customerMessageIds.size());

        // 收集执行完成的客服消息ID
        List<Long> finishedIdList = new ArrayList<>();
        
        try {
            // 遍历执行setPack方法
            for (CustomerBatchSetPackDTO dto : batchSetPackList) {
                try {
                    LOCAL_LOG.info("正在处理客服消息ID: {}, 人群包ID: {}", dto.getCustomerMessageId(), dto.getPackId());
                    BaseResult result = setPack(dto.getCustomerMessageId(), dto.getPackId());
                    if (result.getSuccess()) {
                        LOCAL_LOG.info("客服消息ID: {} 设置人群包成功", dto.getCustomerMessageId());
                    } else {
                        LOCAL_LOG.error("客服消息ID: {} 设置人群包失败: {}", dto.getCustomerMessageId(), result.getMessage());
                    }
                    finishedIdList.add(dto.getCustomerMessageId());

                    TimeUnit.SECONDS.sleep(30);
                } catch (Exception e) {
                    LOCAL_LOG.error("客服消息ID: {} 设置人群包异常", dto.getCustomerMessageId(), e);
                }
            }
        } catch (Exception e) {
            LOCAL_LOG.error("批量设置人群包整体异常", e);
            // 计算未执行成功的客服消息ID
            List<Long> failedIdList = customerMessageIds.stream()
                    .filter(id -> !finishedIdList.contains(id))
                    .collect(Collectors.toList());
            
            if (!failedIdList.isEmpty()) {
                // 将未执行成功的客服消息状态设置为3（链接生成异常）
                updateEntity.setSendStatus(3);
                this.update(updateEntity, new UpdateWrapper<CustomerGraphics>().in("id", failedIdList));
                LOCAL_LOG.info("已将 {} 条客服消息状态设置为链接生成异常", failedIdList.size());
            }
        }

        LOCAL_LOG.info("批量设置人群包完成，成功数量: {}, 总数量: {}", finishedIdList.size(), batchSetPackList.size());
    }

    /**
     * 发送客服消息
     * @param integrationRequestDTO 请求DTO
     */
    @Override
    public void sendCustomerMessage(IntegrationRequestDTO integrationRequestDTO) {

        Date requestDate = integrationRequestDTO.getRequestDate();

        long currentTimestamp = System.currentTimeMillis();
        //获取待发送的开启客服
        List<CustomerGraphics> customerGraphicsList = list(new QueryWrapper<CustomerGraphics>().eq(CustomerGraphics.COL_POST_TIME, requestDate).eq(CustomerGraphics.COL_SEND_STATUS,CustomerCommonConstant.SEND_STATUS_ACTIVE));

        if(!customerGraphicsList.isEmpty()){

            List<CrowdPackage> crowdPackageList = crowdPackageService.list();

            Map<Long, CrowdPackage> crowdPackageMap = crowdPackageList.stream().collect(Collectors.toMap(CrowdPackage::getId, a -> a));

            for (CustomerGraphics customerGraphics : customerGraphicsList) {

                try {
                    LOCAL_LOG.info("{} start sendCustomerMessage:{}",customerGraphics.getId(),System.currentTimeMillis());
                    //人群包id
                    Long packId = customerGraphics.getPackId();

                    CrowdPackage crowdPackage = crowdPackageMap.get(packId);
                    if (crowdPackage == null) {
                        LOCAL_LOG.warn("crowd package id {} not exist", packId);
                        continue;
                    }
                    //活跃时间限制窗口
                    Long activeTimeMax = crowdPackage.getActiveTimeMax();
                    Long activeTimeMin = crowdPackage.getActiveTimeMin();
                    Integer followTimeMin = crowdPackage.getFollowTimeMin();
                    Integer followTimeMax = crowdPackage.getFollowTimeMax();

                    //根据人群包找人,并按appId分组
                    List<CrowdPackageUserVO> userList = registerUserEntityService.getCurrentInPackUserList(packId, false);

                    //1.活跃时间判断，2.用户去重
                    Map<String, List<CrowdPackageUserVO>> appidUserMap = userList.stream()
                            .filter(a -> a.getLastActive() != null &&
                                    (activeTimeMin == null || ((currentTimestamp - a.getLastActive().getTime()) >= activeTimeMin)) &&
                                    (activeTimeMax == null || ((currentTimestamp - a.getLastActive().getTime()) < activeTimeMax)) &&
                                    (followTimeMin == null || (a.getDateDiff() >= followTimeMin))&&
                                    (followTimeMax == null || (a.getDateDiff() <= followTimeMax)))
                            .collect(Collectors.groupingBy(CrowdPackageUserVO::getAppId, Collectors.collectingAndThen(toCollection(() -> new TreeSet<>(Comparator.comparing(CrowdPackageUserVO::getOpenId))), ArrayList::new)));

                    //根据客服id找不同公众号的链接,并按appId分组
                    List<ReferralEntityVo> referralList = referralEntityService.findReferralByCustomerGraphicsId(customerGraphics.getId());
                    // 提取出其中的自定义链接（因为自定义链接是不会在生成链接的时候跟随公众号生成的，所有公众号用的都是一样的）
                    List<ReferralEntityVo> customLinkList = referralList.stream()
                            .filter(v -> StringUtils.isBlank(v.getAppid()) && StringUtils.isNotBlank(v.getReferral()))
                            .filter(v -> v.getNewsType() != null && v.getNewsType() == -1)
                            .collect(Collectors.toList());
                    // 非自定义链接
                    Map<String, List<ReferralEntityVo>> referralMap = referralList
                            // 允许小程序类型的不设置referral字段的值
                            .stream().filter(a ->
                                    // 过滤掉：appId 为空 或者（非小程序类型且链接为空） 的referral记录
                                    StringUtils.isNotBlank(a.getAppid())
                                            && (StringUtils.isNotBlank(a.getReferral()) || CustomerCommonConstant.CUSTOMER_TYPE_VALUE_MINI_PROGRAM.equals(customerGraphics.getType()) ))
                            .collect(Collectors.groupingBy(ReferralEntityVo::getAppid));
                    //循环该人群包下的所有素材
                    int singleUserCount = 0;
                    for (Map.Entry<String, List<CrowdPackageUserVO>> usersEntry : appidUserMap.entrySet()) {

                            String appid = usersEntry.getKey();
                            try {  //获取该号的链接实体
                                List<ReferralEntityVo> referralEntityVo = referralMap.getOrDefault(appid, CollectionUtil.newArrayList());
                                if (CollectionUtil.isNotEmpty(customLinkList)) {
                                    referralEntityVo.addAll(customLinkList);
                                }
                                //获取该号的openid
                                List<CrowdPackageUserVO> packageUserVo = usersEntry.getValue();
                                singleUserCount += packageUserVo.size();
                                weChatService.sendCustomerMessage(appid,customerGraphics,packageUserVo,referralEntityVo);

                            }catch (Exception e){
                                LOCAL_LOG.error("{} send CustomerMessage failed for {}",appid,e.getStackTrace()[0]);
                            }

                    }
                    LOCAL_LOG.info("id {} has user list size {}", customerGraphics.getId(), singleUserCount);
                    //客服状态修改
                    customerGraphics.setSendStatus(CustomerCommonConstant.SEND_STATUS_FINISHED);
                    LOCAL_LOG.info("{} finished:{}",customerGraphics.getId(),System.currentTimeMillis());
                } catch (Exception e) {
                    LOCAL_LOG.error("send CustomerMessage failed for {}", customerGraphics.getId(), e);
                }
            }
            updateBatchById(customerGraphicsList);

        }
    }

    private BaseResult checkExtendBook(String storeType, CrowdPackageConditionMatch storeConditionMatch) {
        // 判断有没有选书城条件
        if (ObjectUtil.isNull(storeConditionMatch)) {
            // 没有的话不符合条件，返回
            return new BaseResult().error("所选人群包未设置书城条件");
        }
        String operatorExpression = storeConditionMatch.getOperatorExpression();
        // 判断选了几个书城，如果超过1个，则不符合条件
        if (operatorExpression.contains(CrowdPackageCommonConstant.COMMA_SEPARATOR)) {
            // 只要表达式包含了逗号，就说明选了不止一个书城条件
            return new BaseResult().error("所选人群包设置的书城条件超过1个");
        }
        // 确定只选了一个书城，判断设置的书是不是这个书城里的
        LOCAL_LOG.info("原始书城条件表达式: {}", operatorExpression);
        operatorExpression = "where " + operatorExpression.replace(CrowdPackageCommonConstant.CONDITION_STORE_ID, "id");
        LOCAL_LOG.info("新的书城条件表达式: {}", operatorExpression);
        String storeTypeByExpression = kanbanCommonMapper.getStoreTypeByExpression(operatorExpression);
        LOCAL_LOG.info("表达式获取到的书城: {}", storeTypeByExpression);

        if (storeType.equals(storeTypeByExpression)) {
            return new BaseResult().success();
        }
        return new BaseResult().error("所选书籍不属于人群包指定书城");

    }


}

