package com.yaoyaozw.customer.components;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.yaoyaozw.customer.common.R;
import com.yaoyaozw.customer.constants.ApiResultConstant;
import com.yaoyaozw.customer.constants.CustomerCommonConstant;
import com.yaoyaozw.customer.constants.CustomerMaterialConstant;
import com.yaoyaozw.customer.dto.follow.FollowReplySaveDTO;
import com.yaoyaozw.customer.dto.follow.NewsMessageReplyEntity;
import com.yaoyaozw.customer.entity.*;
import com.yaoyaozw.customer.exception.BaseException;
import com.yaoyaozw.customer.factory.MpReferralFactory;
import com.yaoyaozw.customer.feigns.ReferralFeignClient;
import com.yaoyaozw.customer.handler.MpReferralHandler;
import com.yaoyaozw.customer.service.AuthorizerExpandInfoService;
import com.yaoyaozw.customer.service.AuthorizerInfoService;
import com.yaoyaozw.customer.service.CompanyAcquisitionLinkService;
import com.yaoyaozw.customer.service.ReferralEntityService;
import com.yaoyaozw.customer.utils.TencentCustomerUtil;
import com.yaoyaozw.customer.vo.AuthInfoVO;
import com.yaoyaozw.customer.vo.TencentMediaResponseVO;
import com.yaoyaozw.customer.vo.follow.FollowReplyCopyResultVO;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import static com.yaoyaozw.customer.constants.CustomerCommonConstant.*;
import static com.yaoyaozw.customer.constants.CustomerCommonConstant.FROM_APPID;

/**
 * @author darker
 * @date 2023/3/14 14:36
 */
@Component
public class FollowReplyComponent {

    private final static Logger localLog = LoggerFactory.getLogger(FollowReplyComponent.class);

    @Autowired
    private ReferralEntityService referralEntityService;
    @Autowired
    private TencentCustomerUtil tencentCustomerUtil;
    @Autowired
    private ReferralFeignClient referralFeignClient;
    @Autowired
    private SnowflakeComponent snowflakeComponent;
    @Autowired
    private RedisTemplate<String, Object> redisTemplate;
    @Resource
    private CompanyAcquisitionLinkService companyAcquisitionLinkService;
    @Resource
    private AuthorizerExpandInfoService authorizerExpandInfoService;
    @Resource
    private AuthorizerInfoService authorizerInfoService;
    @Resource
    private MpReferralFactory mpReferralFactory;

    /**
     * 获取链接实体
     * @param referralBody 链接参数
     * @return 转化之后的链接实体
     */
    public ReferralEntity getCreateReferralEntity(CommonReferralBody referralBody) {

        ReferralEntity referralEntity = new ReferralEntity();

        // 赋值链接参数
        BeanUtil.copyProperties(referralBody, referralEntity);

        Integer newsType = referralEntity.getNewsType();
        // 判断是否需要获取书城链接
        if (CustomerCommonConstant.REMOTE_LINK_NEWS_TYPE_LIST.contains(referralBody.getNewsType())) {
            // 处理链接名称
            if (CustomerCommonConstant.BOOK_NEWS_TYPE.equals(newsType) || CustomerCommonConstant.ACTIVITY_NEWS_TYPE.equals(newsType)) {
                SimpleDateFormat format = new SimpleDateFormat("yyyy/MM/dd");
                String dateStr = format.format(new Date());

                handleReferralName(dateStr, referralBody.getAccountName(), referralEntity);
            }

            // 获取链接
            doGetReferral(referralEntity);
        }

        return referralEntity;
    }

    public ReferralEntity getCreateReferralEntityMp(CommonReferralBody referralBody) {
        ReferralEntity referralEntity = new ReferralEntity();
        // 赋值链接参数
        BeanUtil.copyProperties(referralBody, referralEntity);

        Integer newsType = referralEntity.getNewsType();
        // 判断是否需要获取书城链接
        if (CustomerCommonConstant.REMOTE_LINK_NEWS_TYPE_LIST.contains(referralBody.getNewsType())) {
            // 处理链接名称
            if (CustomerCommonConstant.BOOK_NEWS_TYPE.equals(newsType) || CustomerCommonConstant.ACTIVITY_NEWS_TYPE.equals(newsType)) {
                SimpleDateFormat format = new SimpleDateFormat("yyyy/MM/dd");
                String dateStr = format.format(new Date());
                handleReferralName(dateStr, referralBody.getAccountName(), referralEntity);
            }
            if (StringUtils.isBlank(referralBody.getStoreType()) && referralBody.getInfoId() != null) {
                AuthorizerInfo authorizerInfo = authorizerInfoService.getById(referralBody.getInfoId());
                if (authorizerInfo != null) {
                    referralBody.setStoreType(authorizerInfo.getStoreType());
                }
            }
            MpReferralHandler handler = mpReferralFactory.getHandler(referralBody.getStoreType());
            if (handler == null) {
                throw new BaseException("不支持小程序的书城：" + referralBody.getStoreType());
            }
            referralEntity = handler.queryMpReferral(referralEntity);
        }
        return referralEntity;
    }

    /**
     * 获取复用时的链接数据
     *
     * @param sourceAccountName 源公众号名称
     * @param dateStr 日期
     * @param authInfoVo 目标公众号
     * @param sourceReferral 链接参数
     * @return {@link ReferralEntity}
     */
    public ReferralEntity getCopyReferralEntity(String sourceAccountName, String dateStr, AuthInfoVO authInfoVo, ReferralEntity sourceReferral) {
        ReferralEntity targetReferral = new ReferralEntity();
        BeanUtil.copyProperties(sourceReferral, targetReferral,
                "id", "accountId", "name", "referral", "materialGraphicsId", "infoId");
        // 为参数设置公众号相关参数
        targetReferral.setAccountId(authInfoVo.getAccountId());
        targetReferral.setInfoId(authInfoVo.getId());
        String textContent = targetReferral.getTextContent();
        if (StringUtils.isNotEmpty(textContent)) {
            targetReferral.setTextContent(textContent.replace(sourceAccountName, authInfoVo.getAccountName()));
        }

        Integer newsType = targetReferral.getNewsType();
        if (CustomerCommonConstant.REMOTE_LINK_NEWS_TYPE_LIST.contains(newsType)) {
            // 需要获取第三方链接的类型
            if (CustomerCommonConstant.USUAL_LINK_NEWS_TYPE.equals(newsType)) {
                // 常用链接延用name
                targetReferral.setName(sourceReferral.getName());
            } else {
                // 书籍、活动 类型, 重新构造name
                handleReferralName(dateStr, authInfoVo.getAccountName(), targetReferral);
            }
            doGetReferral(targetReferral);
        } else if (newsType.equals(-1)) {
            // 自定义链接，校验是不是获客链接，如果是的，更换为目标公众号的获客链接
            targetReferral.setReferral(companyAcquisitionLinkService.checkAndSearchTargetReferral(sourceReferral.getReferral(), authInfoVo.getId()));
        }

        return targetReferral;

    }

    /**
     * 素材复用到目标公众号
     *
     * @param targetAuth         目标公众号
     * @param sourceMaterialList 源素材列表
     * @return 执行结果
     */
    public FollowReplyCopyResultVO copyMaterialToTarget(String sourceAccountName, AuthInfoVO targetAuth, List<CustomerFollowReply> sourceMaterialList) {
        FollowReplyCopyResultVO result = new FollowReplyCopyResultVO();
        result.setHasError(false);

        List<CustomerFollowReply> finalMaterialList = new ArrayList<>(sourceMaterialList.size());
        List<CustomerFollowReplyMultiNews> finalMultiNewsList = new ArrayList<>();
        List<ReferralEntity> finalReferralList = new ArrayList<>();

        SimpleDateFormat format = new SimpleDateFormat("yyyy/MM/dd");
        String dateStr = format.format(new Date());

        int idx = 1;
        for (CustomerFollowReply sourceMaterial : sourceMaterialList) {
            localLog.info("素材处理进度: {}/{}", idx++, sourceMaterialList.size());
            CustomerFollowReply entity = new CustomerFollowReply();
            int status = 1;

            // 主体赋值
            entity.setId(snowflakeComponent.snowflakeId());
            entity.setAppid(targetAuth.getAppId());
            entity.setName(sourceMaterial.getName());
            entity.setType(sourceMaterial.getType());
            entity.setOriginMediaUrl(sourceMaterial.getOriginMediaUrl());
            entity.setExtendTitle(sourceMaterial.getExtendTitle());
            entity.setExtendDesc(sourceMaterial.getExtendDesc());
            entity.setSort(sourceMaterial.getSort());
            entity.setMpTitle(sourceMaterial.getMpTitle());
            entity.setMpAppId(sourceMaterial.getMpAppId());
            entity.setMpPath(sourceMaterial.getMpPath());

            if (CustomerMaterialConstant.needUpload(entity.getType())) {
                // 向腾讯后台上传素材文件
                TencentMediaResponseVO uploadResult = tencentCustomerUtil.uploadTencentMedia(targetAuth.getAppId(), entity.getOriginMediaUrl(), entity.getType());
                if (StringUtils.isNotEmpty(uploadResult.getErrmsg())) {
                    localLog.error("公众号: {} 上传素材失败 {}", targetAuth.getAccountName(), uploadResult.getErrmsg());
                }
                entity.setTxMediaUrl(uploadResult.getUrl());
                entity.setTxMediaId(uploadResult.getMedia_id());

                // 判断需不需要处理小程序
                try {
                    handleMp(dateStr, targetAuth, entity, sourceMaterial);
                    finalMaterialList.add(entity);
                } catch (Exception e) {
                    status = 0;
                    result.setHasError(true);
                }
                entity.setStatus(status);
                continue;
            }

            if (!CustomerMaterialConstant.TENCENT_MEDIA_TYPE_MULTI_NEWS.equals(sourceMaterial.getType())) {
                // 非多图文类型 处理 图文、文本 链接
                List<ReferralEntity> referralEntityList = sourceMaterial.getReferralEntityList();
                if (CollectionUtil.isNotEmpty(referralEntityList)) {
                    List<ReferralEntity> singleMaterialReferralList = new ArrayList<>(referralEntityList.size());
                    for (ReferralEntity sourceReferral : referralEntityList) {
                        try {
                            ReferralEntity finalReferral = getCopyReferralEntity(sourceAccountName, dateStr, targetAuth, sourceReferral);
                            finalReferral.setMaterialGraphicsId(entity.getId());
                            singleMaterialReferralList.add(finalReferral);
                            finalReferralList.add(finalReferral);
                        } catch (Exception e) {
                            // 获取链接出现异常, 修改状态, 修改标识
                            status = 0;
                            result.setHasError(true);
                        }
                    }
                    if (CollectionUtil.isNotEmpty(singleMaterialReferralList)) {
                        // 处理链接的后续操作
                        if (CustomerMaterialConstant.TENCENT_MEDIA_TYPE_NEWS.equals(entity.getType())) {
                            // 图文, 将链接处理结果设置到主题表中
                            ReferralEntity referralEntity = singleMaterialReferralList.get(0);
                            entity.setSourceUrl(referralEntity.getReferral());
                        } else if (CustomerMaterialConstant.TENCENT_MEDIA_TYPE_TEXT.equals(entity.getType())) {
                            // 文本, 构造H5
                            StringBuilder builder = new StringBuilder();
                            for (ReferralEntity referralEntity : singleMaterialReferralList) {
                                builder.append(structH5Line(false, referralEntity)).append("\n\n");
                            }
                            entity.setContent(builder.toString().trim());

                        }
                    }

                }
                entity.setStatus(status);
                finalMaterialList.add(entity);
            } else {
                // 多图文
                entity.setStatus(status);
                redisTemplate.opsForHash().delete(CustomerMaterialConstant.FOLLOW_REPLY_MULTI_NEWS_CACHE_KEY, targetAuth.getAppId());
                List<CustomerFollowReplyMultiNews> sourceMultiNewsList = sourceMaterial.getMultiNewsList();
                for (CustomerFollowReplyMultiNews sourceMultiNews : sourceMultiNewsList) {
                    ReferralEntity sourceReferralEntity = sourceMultiNews.getReferralEntity();
                    ReferralEntity targetReferralEntity = getCopyReferralEntity(sourceAccountName, dateStr, targetAuth, sourceReferralEntity);

                    CustomerFollowReplyMultiNews targetMultiNews = new CustomerFollowReplyMultiNews();
                    BeanUtil.copyProperties(sourceMultiNews, targetMultiNews, "id", "replyId", "linkUrl", "referralEntity");
                    targetMultiNews.setId(snowflakeComponent.snowflakeId());
                    targetMultiNews.setReplyId(entity.getId());
                    targetMultiNews.setLinkUrl(targetReferralEntity.getReferral());

                    targetReferralEntity.setMaterialGraphicsId(targetMultiNews.getId());
                    finalMultiNewsList.add(targetMultiNews);
                    finalReferralList.add(targetReferralEntity);
                }
                finalMaterialList.add(entity);
            }

        }

        result.setMaterialList(finalMaterialList);
        result.setReferralEntityList(finalReferralList);
        result.setMultiNewsList(finalMultiNewsList);
        if (CollectionUtil.isNotEmpty(finalMultiNewsList)) {
            List<NewsMessageReplyEntity> cacheEntityList = JSONUtil.toList(JSONUtil.parseArray(finalMultiNewsList), NewsMessageReplyEntity.class);
            redisTemplate.opsForHash().put(CustomerMaterialConstant.FOLLOW_REPLY_MULTI_NEWS_CACHE_KEY, targetAuth.getAppId(), JSONUtil.toJsonStr(cacheEntityList));
        }
        return result;

    }

    private void handleMp(String dateStr, AuthInfoVO targetAuth, CustomerFollowReply neoEntity, CustomerFollowReply sourceMaterial) {
        if (!CUSTOMER_TYPE_VALUE_MINI_PROGRAM.equals(neoEntity.getType())) {
            return;
        }
        List<ReferralEntity> referralEntityList = sourceMaterial.getReferralEntityList();
        if (CollectionUtil.isEmpty(referralEntityList)) {
            // 不具备处理条件
            return;
        }
        ReferralEntity sourceReferral = referralEntityList.get(0);
        // 如果参数传入的标题只是在图文上，则延用过来
        if (StringUtils.isBlank(neoEntity.getMpTitle()) && StringUtils.isNotBlank(neoEntity.getExtendTitle())) {
            neoEntity.setMpTitle(neoEntity.getExtendTitle());
        }
        if (StringUtils.isBlank(neoEntity.getMpTitle())) {
            throw new BaseException("无法匹配小程序发文的标题内容");
        }
        // 重新获取链接
        ReferralEntity targetReferral = new ReferralEntity();
        try {
            BeanUtil.copyProperties(sourceReferral, targetReferral, "id", "accountId", "name", "referral", "materialGraphicsId", "infoId");
            // 为参数设置公众号相关参数
            targetReferral.setInfoId(targetAuth.getId());
            targetReferral.setStoreType(targetAuth.getStoreType());
            Integer newsType = targetReferral.getNewsType();
            if (CustomerCommonConstant.USUAL_LINK_NEWS_TYPE.equals(newsType)) {
                // 常用链接延用name
                targetReferral.setName(sourceReferral.getName());
            } else {
                // 书籍、活动 类型, 重新构造name
                handleReferralName(dateStr, targetAuth.getAccountName(), targetReferral);
            }
            MpReferralHandler handler = mpReferralFactory.getHandler(targetReferral.getStoreType());
            if (handler == null) {
                throw new BaseException("不支持小程序的书城：" + targetReferral.getStoreType());
            }
            targetReferral = handler.queryMpReferral(targetReferral);
        } catch (Exception e) {
            throw new RuntimeException("获取链接异常: " + e.getMessage());
        }
        targetReferral.setMaterialGraphicsId(neoEntity.getId());
        neoEntity.setMpPath(targetReferral.getReferral());
        neoEntity.setMpAppId(targetReferral.getMpAppId());
        neoEntity.setContent("");
        // 保存链接数据
        if (ObjectUtil.isNull(targetReferral.getId())) {
            referralEntityService.save(targetReferral);
        } else {
            referralEntityService.updateById(targetReferral);
        }
    }


    /**
     * 重新排序和保存文本子素材
     *
     * @param referralList 文本子素材列表
     * @return {@link String}
     */
    public String reSortAndSaveTextItem(List<ReferralEntity> referralList) {
        // 遍历进行排序，并且构造H5
        int idx = 1;
        StringBuilder builder = new StringBuilder();
        for (ReferralEntity referral : referralList) {
            referral.setSort(idx);
            builder.append(structH5Line(false, referral));

            if (idx < referralList.size()) {
                builder.append("\n\n");
                idx++;
            }
        }
        // 保存排序之后的referral列表
        referralEntityService.updateBatchById(referralList);

        return builder.toString();
    }

    public void contractH5ContentBatch(List<ReferralEntity> referralEntityList) {
        referralEntityList.forEach(item -> item.setH5Content(structH5Line(true, item)));
    }

    /**
     * 构造 h5
     *
     * @param referralEntity 链接实体
     * @return {@link String}
     */
    private String structH5Line(Boolean needStyle, ReferralEntity referralEntity) {
        Integer newsType = referralEntity.getNewsType();
        // 文本类型替换h5链接
        String context = null;
        if (CustomerCommonConstant.REPLACE_LINK_NEWS_TYPE_LIST.contains(newsType)) {
            context = CustomerCommonConstant.CUSTOMER_TEXT_LINK_TEMPLATE.replace(CustomerCommonConstant.CUSTOMER_TEXT_CONTENT_PLACEHOLDER, referralEntity.getTextContent());
            context = context.replace(CustomerCommonConstant.CUSTOMER_TEXT_URL_PLACEHOLDER, referralEntity.getReferral());
            if (!needStyle) {
                context = context.replace(CustomerCommonConstant.H5_STYLE_CODE, "");
            }
        } else if (CustomerCommonConstant.COMMON_NEWS_TYPE_LIST.contains(newsType)){
            context = referralEntity.getTextContent();
        }

        return context;
    }

    /**
     * 处理链接名称
     * @param dateStr 处理日期
     * @param accountName 公众号名称
     * @param referral 链接参数实体
     */
    private void handleReferralName(String dateStr, String accountName, ReferralEntity referral) {
        Integer newsType = referral.getNewsType();
        // 书籍、活动 类型, 重新构造name
        String name = CustomerCommonConstant.getLinkNameModel(newsType, referral.getConfigName());
        String newsTypeName = CustomerCommonConstant.getNewsTypeName(newsType);
        name = name.replace("{newsType}", newsTypeName).replace("{accountNickName}", accountName)
                .replace("{storeType}", referral.getStoreTypeName()).replace("{currentDate}", dateStr);

        String bookName = referral.getBookName();
        if (StringUtils.isNotBlank(bookName)) {
            bookName = bookName.substring(0, Math.min(8, bookName.length()));
        }
        if (newsType.equals(CustomerCommonConstant.BOOK_NEWS_TYPE)) {
            // 系统-客服-{newsType}-{accountNickName}-{storeType}-{currentDate}-{bookName}
            name = name.replace("{bookName}", bookName);
        } else if (newsType.equals(CustomerCommonConstant.ACTIVITY_NEWS_TYPE)) {
            // 系统-客服-{newsType}-{accountNickName}-{storeType}-{currentDate}-充{recharge}送{gift}
            if (StringUtils.isNotEmpty(referral.getConfigName())) {
                name = name.replace("{configName}", referral.getConfigName());
            } else {
                name = name.replace("{recharge}", referral.getRechargeAmount().stripTrailingZeros().toPlainString()).replace("{gift}", referral.getGiftAmount().toString());
            }
        }
        name = name.replace("客服", "关回");
        referral.setName(name + "-" +System.currentTimeMillis() % 100000);
    }

    /**
     * 获取链接
     *
     * @param referralEntity 链接参数
     */
    private void doGetReferral(ReferralEntity referralEntity) {
        // 获取链接
        R r = referralFeignClient.productReferral(referralEntity);
        if (!r.getCode().equals(ApiResultConstant.SUCCESS_CODE)) {
            localLog.warn("获取链接异常, 异常信息: {}, 参数: {}", r.getMessage(), referralEntity);
            throw new RuntimeException(r.getMessage());
        }
        String res = r.getData("storeReferral", new TypeReference<String>() {});
        JSONObject jsonObject1 = JSON.parseObject(res);
        String referral = jsonObject1.getString("referral");
        referralEntity.setPromoteId(jsonObject1.getString("promoteId"));
        referralEntity.setReferral(referral);
    }

}
