package com.yaoyaozw.customer.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yaoyaozw.customer.common.GenericsResult;
import com.yaoyaozw.customer.components.SnowflakeComponent;
import com.yaoyaozw.customer.components.TokenManager;
import com.yaoyaozw.customer.constants.CustomerCommonConstant;
import com.yaoyaozw.customer.dto.customer.CustomerMessageTextSaveDTO;
import com.yaoyaozw.customer.dto.customer.CustomerReferralDTO;
import com.yaoyaozw.customer.entity.CustomerGraphics;
import com.yaoyaozw.customer.entity.ReferralEntity;
import com.yaoyaozw.customer.mapper.CustomerGraphicsMapper;
import com.yaoyaozw.customer.service.CustomerGraphicsTextService;
import com.yaoyaozw.customer.service.ReferralEntityService;
import com.yaoyaozw.customer.vo.customer.CustomerContentVO;
import com.yaoyaozw.customer.vo.customer.CustomerGraphicTextDetailVO;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import java.util.ArrayList;
import java.util.List;

/**
 * @author darker
 * @date 2022/10/12 11:49
 */
@Service
public class CustomerGraphicsTextServiceImpl extends ServiceImpl<CustomerGraphicsMapper, CustomerGraphics> implements CustomerGraphicsTextService {

    private final static Logger LOCAL_LOG = LoggerFactory.getLogger(CustomerGraphicsTextServiceImpl.class);

    @Autowired
    private TokenManager tokenManager;
    @Autowired
    private SnowflakeComponent snowflakeComponent;
    @Autowired
    private ReferralEntityService referralEntityService;

    @Override
    public GenericsResult<String> insertCustomerMessage(CustomerMessageTextSaveDTO saveDto) {
        // 主体数据
        LOCAL_LOG.info("处理主体数据");
        CustomerGraphics customerGraphics = new CustomerGraphics();
        BeanUtil.copyProperties(saveDto, customerGraphics);
        customerGraphics.initOperateInfo(tokenManager.getUserIdFromToken(), ObjectUtil.isNull(saveDto.getId()));
        if (ObjectUtil.isNull(customerGraphics.getId())) {
            long id = snowflakeComponent.snowflakeId();
            customerGraphics.setId(id);
        }
        super.saveOrUpdate(customerGraphics);

        return new GenericsResult<>(customerGraphics.getId().toString());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public GenericsResult<CustomerGraphicTextDetailVO> insertCustomerContent(CustomerReferralDTO referralDto) {

        System.out.println(referralDto);

        // 处理活动数据
        ReferralEntity referralEntity = new ReferralEntity();
        BeanUtil.copyProperties(referralDto, referralEntity);
        // 获取name模板
        String nameModel = CustomerCommonConstant.getLinkNameModel(referralEntity.getNewsType());
        LOCAL_LOG.info("获取name模板: {}", nameModel);
        if (StringUtils.isNotBlank(nameModel)) {
            referralEntity.setName(nameModel);
        }
        // 保存链接数据
        referralEntityService.saveOrUpdate(referralEntity);

        return getCustomerTextDetail(referralDto.getMaterialGraphicsId());
    }

    @Override
    public GenericsResult<CustomerGraphicTextDetailVO> getCustomerTextDetail(Long id) {
        // 查询主体数据
        CustomerGraphics customerGraphics = super.getById(id);
        if (ObjectUtil.isNull(customerGraphics)) {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            return new GenericsResult<>(false, "找不到主体数据");
        }
        // 赋值主体数据
        CustomerGraphicTextDetailVO detailVO = new CustomerGraphicTextDetailVO();
        BeanUtil.copyProperties(customerGraphics, detailVO);
        detailVO.setPostTime(customerGraphics.getPostTimeStr());

        // 合并链接中的h5代码
        List<ReferralEntity> referralEntityList = referralEntityService.list(new QueryWrapper<ReferralEntity>().eq("material_graphics_id", id).isNull("account_id").orderByAsc("gmt_create"));

        List<CustomerContentVO> contentList = new ArrayList<>();
        // 位置计数器
        StringBuilder mainH5Content = new StringBuilder();

        LOCAL_LOG.info("根据链接重新生成H5");
        int idx = 1;
        int replaceLinkTypeNum = 0;
        for (ReferralEntity item : referralEntityList) {
            item.setSort(idx);
            CustomerContentVO customerContentVO = new CustomerContentVO();
            // 设置主键id和h5代码段
            customerContentVO.setId(item.getId());
            String context = null;
            if (CustomerCommonConstant.LINK_NEWS_TYPE_LIST.contains(item.getNewsType())) {
                context = CustomerCommonConstant.CUSTOMER_TEXT_LINK_TEMPLATE
                        .replace(CustomerCommonConstant.CUSTOMER_TEXT_CONTENT_PLACEHOLDER, item.getTextContent())
                        .replace(CustomerCommonConstant.CUSTOMER_TEXT_URL_PLACEHOLDER, CustomerCommonConstant.CUSTOMER_TEXT_URL_PLACEHOLDER + idx);
            } else if (CustomerCommonConstant.COMMON_NEWS_TYPE_LIST.contains(item.getNewsType())){
                context = item.getTextContent();
            }
            if (ObjectUtil.isNotNull(context)) {
                mainH5Content.append(context);
                if (idx != referralEntityList.size()) {
                    mainH5Content.append("\n").append("\n");
                }
                // 将链接中的占位符换成#
                customerContentVO.setContent(context.replace(CustomerCommonConstant.CUSTOMER_TEXT_URL_PLACEHOLDER+idx, "#"));
            }

            // 赋值链接信息
            CustomerReferralDTO customerReferralDto = new CustomerReferralDTO();
            BeanUtil.copyProperties(item, customerReferralDto);

            customerContentVO.setCustomerReferralDto(customerReferralDto);
            contentList.add(customerContentVO);

            // 统计referralSize
            if (CustomerCommonConstant.REPLACE_LINK_NEWS_TYPE_LIST.contains(item.getNewsType())) {
                replaceLinkTypeNum += 1;
            }
            idx += 1;
        }
        customerGraphics.setContent(mainH5Content.toString());
        customerGraphics.setReferralSize(replaceLinkTypeNum);
        detailVO.setContentList(contentList);
        // 更新连接表
        LOCAL_LOG.info("更新连接表");
        if (CollectionUtil.isNotEmpty(referralEntityList)) {
            referralEntityService.updateBatchById(referralEntityList);
        }
        // 更新主表
        LOCAL_LOG.info("更新主表");
        customerGraphics.initOperateInfo(tokenManager.getUserIdFromToken(), false);
        super.updateById(customerGraphics);

        return new GenericsResult<>(detailVO);
    }

    @Override
    public GenericsResult<CustomerGraphicTextDetailVO> removeCustomerContent(Long contentId) {
        ReferralEntity referralEntity = referralEntityService.getById(contentId);
        if (ObjectUtil.isNull(referralEntity)) {
            return new GenericsResult<>(false, "找不到主体数据");
        }
        boolean result = referralEntityService.removeById(contentId);
        if (result) {
            return getCustomerTextDetail(referralEntity.getMaterialGraphicsId());
        }
        return new GenericsResult<>(false, "删除失败");
    }

}
