package com.yaoyaozw.customer.components;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.yaoyaozw.customer.common.R;
import com.yaoyaozw.customer.constants.ApiResultConstant;
import com.yaoyaozw.customer.constants.CrowdPackageCommonConstant;
import com.yaoyaozw.customer.constants.CustomerCommonConstant;
import com.yaoyaozw.customer.dto.customer.CustomerMessageSaveDTO;
import com.yaoyaozw.customer.entity.CustomerGraphics;
import com.yaoyaozw.customer.entity.ReferralEntity;
import com.yaoyaozw.customer.entity.RegisterUserEntity;
import com.yaoyaozw.customer.feigns.ReferralFeignClient;
import com.yaoyaozw.customer.mapper.MaterialCommonMapper;
import com.yaoyaozw.customer.service.CrowdPackageConditionMatchService;
import com.yaoyaozw.customer.service.ReferralEntityService;
import com.yaoyaozw.customer.service.RegisterUserEntityService;
import com.yaoyaozw.customer.vo.AuthInfoVO;
import com.yaoyaozw.customer.vo.customer.CrowdPackageUserVO;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author darker
 * @date 2022/9/28 15:16
 */
@Component
public class CustomerServiceCommonAsyncComponent {

    private final static Logger LOCAL_LOG = LoggerFactory.getLogger(CustomerServiceCommonAsyncComponent.class);

    @Autowired
    private RegisterUserEntityService userEntityService;
    @Autowired
    private ReferralEntityService referralEntityService;
    @Autowired
    private CrowdPackageConditionMatchService conditionMatchService;
    @Autowired
    private SnowflakeComponent snowflakeComponent;
    @Autowired
    private CrowdPackageConditionMatchService matchService;
    @Autowired
    private RedisTemplate redisTemplate;
    @Autowired
    private ReferralFeignClient referralFeignClient;
    @Autowired
    private MaterialCommonMapper materialCommonMapper;


    @Async("myExecutor")
    public void addMatchUserIntoPackage(Long packageId, Boolean needRemove) {
        // 获取符合人群包条件的用户列表
        List<CrowdPackageUserVO> userList = matchService.getUserListFromPackage(packageId, null);
        redisTemplate.opsForHash().put(CustomerCommonConstant.CROWD_HUMAN_NUN_REDIS_KEY, packageId.toString(), userList.size());
        long startTime = System.currentTimeMillis();
        String packIdStr = packageId.toString();
        // 筛选当前不在这个人群包的用户
        List<CrowdPackageUserVO> userNotInPackageList = userList.stream()
                // 保留现在不在该人群包的用户
                .filter(item -> StringUtils.isBlank(item.getInPackage()) || !item.getInPackage().contains(packIdStr))
                .peek(item -> {
                    if (StringUtils.isBlank(item.getInPackage())) {
                        item.setInPackage(packIdStr);
                    } else {
                        // 将当前人群包拼在后面
                        item.setInPackage(item.getInPackage() + "," + packIdStr);
                    }
                }).collect(Collectors.toList());

        LOCAL_LOG.info("人群包ID: {} 新增用户 {}个", packIdStr, userNotInPackageList.size());
        if (needRemove) {
            // 当前在人群包内的用户
            List<CrowdPackageUserVO> packCurrentUserList = userEntityService.getCurrentInPackUserList(packageId,true);
            if (CollectionUtil.isNotEmpty(packCurrentUserList)) {
                List<String> userOpenIdList = userList.stream().map(CrowdPackageUserVO::getOpenId).collect(Collectors.toList());
                // 筛选出之前在人群包内，现在不在地用户列表
                List<CrowdPackageUserVO> needRemoveUserList = packCurrentUserList.stream().filter(item -> !userOpenIdList.contains(item.getOpenId())).collect(Collectors.toList());
                removeUnMatchUserFromPackage(packageId, needRemoveUserList);
            }
        }
        if (CollectionUtil.isNotEmpty(userNotInPackageList)) {
            userEntityService.updateBatchById(encapsulateUserEntity(userNotInPackageList));
        }

        long endTime = System.currentTimeMillis();
        LOCAL_LOG.info("异步添加符合条件用户人群包完成, 耗时: {}ms", endTime - startTime);
    }

    @Async("myExecutor")
    public void removeUnMatchUserFromPackage(Long packageId, List<CrowdPackageUserVO> userList) {
        long startTime = System.currentTimeMillis();
        String packIdStr = packageId.toString();
        // 筛选当前不在这个人群包的用户
        List<CrowdPackageUserVO> userRemoveFromPackList = userList.stream()
                // 移除现在在该人群包但不符合当前条件的用户
                .filter(item -> StringUtils.isNotBlank(item.getInPackage()) && item.getInPackage().contains(packIdStr))
                .peek(item -> {
                    String removePackageStr = item.getInPackage().replace("," + packIdStr, "")
                            .replace(packIdStr + ",", "")
                            .replace(packIdStr, "");
                    item.setInPackage(StringUtils.isBlank(removePackageStr) ? "" : removePackageStr);
                }).collect(Collectors.toList());

        LOCAL_LOG.info("人群包ID: {} 移除用户 {}个", packIdStr, userRemoveFromPackList.size());
        if (CollectionUtil.isNotEmpty(userRemoveFromPackList)) {
            userEntityService.updateBatchById(encapsulateUserEntity(userRemoveFromPackList));
        }

        long endTime = System.currentTimeMillis();
        LOCAL_LOG.info("异步删除不符合条件用户所属人群包完成, 耗时: {}ms", endTime - startTime);
    }

    @Async("myExecutor")
    public void getAuthListLink(CustomerGraphics customerMain, ReferralEntity customerReferral) {
        Long packId = customerMain.getPackId();
        LOCAL_LOG.info("开始异步处理客服链接, 客服标题: {}, 人群包主键: {}", customerMain.getName(), packId);

        // 获取符合人群包条件的用户所属的公众号列表
        List<CrowdPackageUserVO> userListFromPackage = conditionMatchService.getUserListFromPackage(packId, null);
        Map<String, List<CrowdPackageUserVO>> storeGroupMap = userListFromPackage.stream().collect(Collectors.groupingBy(CrowdPackageUserVO::getStoreType));
        LOCAL_LOG.info("当前包含书城: {} 个", storeGroupMap.size());

        // 遍历处理书城
        Integer newsType = customerReferral.getNewsType();
        // 一个书城一个书城地处理
        if (newsType.equals(CustomerCommonConstant.BOOK_NEWS_TYPE)) {
            // 是推广链接
            generateExtendLink(storeGroupMap, customerReferral);
        } else if (CustomerCommonConstant.ACTIVITY_NEWS_TYPE.equals(newsType)) {
            // 是活动链接
            generateActivityLink(storeGroupMap, customerReferral);
        } else if (CustomerCommonConstant.USUAL_LINK_NEWS_TYPE.equals(newsType)) {
            // 常用链接
            generateUsualLink(storeGroupMap, customerReferral);
        }


    }

    private List<RegisterUserEntity> encapsulateUserEntity(List<CrowdPackageUserVO> userList) {
        List<RegisterUserEntity> userEntityList = new ArrayList<>();
        for (CrowdPackageUserVO crowdPackageUserVO : userList) {
            RegisterUserEntity registerUserEntity = new RegisterUserEntity();
            registerUserEntity.setId(crowdPackageUserVO.getId());
            registerUserEntity.setInPackage(crowdPackageUserVO.getInPackage());
            userEntityList.add(registerUserEntity);
        }
        return userEntityList;
    }

    private void generateExtendLink(Map<String, List<CrowdPackageUserVO>> storeGroupMap, ReferralEntity customerReferral) {

        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

        storeGroupMap.forEach((storeType, userVoList) -> {
            LOCAL_LOG.info("当前处理书城: {}", storeType);
            // 去重提取公众号
            Set<String> accountSet = userVoList.stream().map(CrowdPackageUserVO::getAccountId).collect(Collectors.toSet());
            List<AuthInfoVO> authInfoList = materialCommonMapper.getAuthInfoList(accountSet);
            // 定义
            List<ReferralEntity> referralEntityList = new ArrayList<>(accountSet.size());
            for (AuthInfoVO authInfoVo : authInfoList) {
                ReferralEntity referralEntity = new ReferralEntity();

                BeanUtil.copyProperties(customerReferral, referralEntity);
                // 重新生成链接主键id
                referralEntity.setId(snowflakeComponent.snowflakeId());
                // TODO: 2022/10/9 获取原文链接
                getCopyReferral(null, authInfoVo, referralEntity);

                referralEntityList.add(referralEntity);
            }

            // 批量新增
            referralEntityService.saveBatch(referralEntityList);
        });
    }

    private void generateActivityLink(Map<String, List<CrowdPackageUserVO>> storeGroupMap, ReferralEntity customerReferral) {

        storeGroupMap.forEach((storeType, userVoList) -> {
            LOCAL_LOG.info("当前处理书城: {}", storeType);
            // 去重提取公众号
            Set<String> accountSet = userVoList.stream().map(CrowdPackageUserVO::getAccountId).collect(Collectors.toSet());

            // 定义
            List<ReferralEntity> referralEntityList = new ArrayList<>(accountSet.size());
            ReferralEntity referralEntity = new ReferralEntity();

            BeanUtil.copyProperties(customerReferral, referralEntity);
            // 重新生成链接主键id
            referralEntity.setId(snowflakeComponent.snowflakeId());
            // TODO: 2022/10/9 获取原文链接

            referralEntityList.add(referralEntity);

            // 批量新增
            referralEntityService.saveBatch(referralEntityList);
        });
    }

    private void generateUsualLink(Map<String, List<CrowdPackageUserVO>> storeGroupMap, ReferralEntity customerReferral) {

        storeGroupMap.forEach((storeType, userVoList) -> {
            LOCAL_LOG.info("当前处理书城: {}", storeType);
            // 去重提取公众号
            Set<String> accountSet = userVoList.stream().map(CrowdPackageUserVO::getAccountId).collect(Collectors.toSet());

            // 定义
            List<ReferralEntity> referralEntityList = new ArrayList<>(accountSet.size());
            ReferralEntity referralEntity = new ReferralEntity();

            BeanUtil.copyProperties(customerReferral, referralEntity);
            // 重新生成链接主键id
            referralEntity.setId(snowflakeComponent.snowflakeId());
            // TODO: 2022/10/9 获取原文链接

            referralEntityList.add(referralEntity);

            // 批量新增
            referralEntityService.saveBatch(referralEntityList);
        });
    }



    private void getCopyReferral(String dateStr,
                                 AuthInfoVO authInfoVo,
                                 ReferralEntity referralEntity) {
        referralEntity.setStoreType(authInfoVo.getStoreType());
        referralEntity.setStoreTypeName(authInfoVo.getStoreTypeName());
        // 非常用链接类型的name需要处理
        if (!CustomerCommonConstant.USUAL_LINK_NEWS_TYPE.equals(referralEntity.getNewsType())) {
            referralEntity.setName(replaceName(referralEntity.getName(), dateStr));
        }
        referralEntity.setAccountId(null);
        referralEntity.setInfoId(authInfoVo.getId());
        R r = referralFeignClient.productReferral(referralEntity);
        if (!r.getCode().equals(ApiResultConstant.SUCCESS_CODE)) {
            throw new RuntimeException(r.getMessage());
        }
        String res = r.getData("storeReferral", new TypeReference<String>() {});
        JSONObject jsonObject1 = JSON.parseObject(res);
        String referral = jsonObject1.getString("referral");
        referralEntity.setPromoteId(jsonObject1.getString("promoteId"));
        referralEntity.setReferral(referral);
        // 系统-客服-{newsType}-{accountNickName}-{storeType}-{currentDate}-{bookName}
        // 系统-客服-{newsType}-{accountNickName}-{storeType}-{currentDate}-充{recharge}送{gift}
    }


    private String replaceName(String materialName, String appNickName) {
        if (StringUtils.isBlank(materialName)) {
            return null;
        }
        if (materialName.contains("-")) {
            String[] split = materialName.split("-");
            String format = new SimpleDateFormat("yyyy/MM/dd").format(new Date());
            split[5] = format;
            split[3] = appNickName;
            return String.join("-", split);
        }
        return materialName;

    }

}
